/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.dockerclient;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import okhttp3.OkHttpClient;
import okhttp3.Request;

public class DockerEnv {
	private static final Logger logger = LoggerFactory.getLogger(DockerEnv.class);

	private static final ObjectMapper mapper = new ObjectMapper();
	private static List<Image> images;
	private static Map<String, String> imageIp = new HashMap<>();
	private static OkHttpClient httpClient;
	private static URI dockerUrl;

	static {
		try {
			init();
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	public String getImageIp(String imageName) {
		return null;
	}

	public static String getIp(String imageName) {
		return imageIp.get(imageName);
	}

	private static void init() throws Exception {
		String home = System.getProperty("user.home");
		File f = new File(home + "/.docker.io.properties");

		String urlString = "unix:///var/run/docker.sock";
		dockerUrl = URI.create("http://localhost:21512/");

		if (f.exists()) {
			logger.info("load docker conf from ~/.docker.io.properties");
			Properties p = new Properties();
			try (InputStream pfile = new FileInputStream(f)) {
				p.load(pfile);
			}
			urlString = p.getProperty("docker.io.url");
		}

		if (urlString.startsWith("unix://")) {
			httpClient = new OkHttpClient.Builder()
					.socketFactory(new UnixDomainSocketFactory(new File(urlString.substring("unix://".length()))))
					.build();
		} else {
			httpClient = new OkHttpClient.Builder().build();
			dockerUrl = URI.create(urlString);
		}

		images = loadImages(new File(""));
		for (Image image : images) {
			String ip = retrieveIp(image.getActualName())
					.orElseGet(() -> retrieveIpFromHostname(image.getName()).orElse(null));
			logger.info("container [{}], actualName: [{}], ip: {}", image.getName(), image.getActualName(), ip);
			imageIp.put(image.getActualName(), ip);
		}
	}

	private static Optional<String> retrieveIpFromHostname(String imageName) {
		String imageAsHostname = imageName.replace('/', '-');

		try {
			return Optional.ofNullable(InetAddress.getByName(imageAsHostname).getHostAddress());
		} catch (Exception e) {
			return Optional.ofNullable(null);
		}
	}

	public static Map<String, String> getImagesMap() {
		return imageIp;
	}

	private static Optional<String> retrieveIp(String imageName) throws Exception {
		imageName = imageName.replace("\\:", "_").replace("/", "_");
		imageName = imageName + "-junit";

		try {
			var req = new Request.Builder().method("GET", null).addHeader("Accept", "application/json")
					.url(dockerUrl.resolve("/containers/" + imageName + "/json").toURL()).build();

			var x = httpClient.newCall(req).execute();
			// NetworkSettings
			// IPAddress
			JsonNode c = mapper.readTree(x.body().bytes());
			logger.debug("{}", c);

			JsonNode networkSettings;
			JsonNode ipAddress;
			if ((networkSettings = c.get("NetworkSettings")) == null
					|| (ipAddress = networkSettings.get("IPAddress")) == null) {
				return Optional.empty();
			}

			return Optional.of(ipAddress.asText());
		} catch (IOException e) {
			if (logger.isDebugEnabled()) {
				logger.debug("Unable to connect to docker", e);
			} else {
				logger.error("Unable to connect to docker: {}", dockerUrl);
			}
			return Optional.empty();
		}

	}

	public List<Image> getImages() {
		return images;
	}

	public static List<Image> loadImages(File root) {
		File ciJson = new File(root.getAbsoluteFile(), "services.json");
		if (!ciJson.exists()) {
			return Collections.emptyList();
		}
		try {
			return mapper.readValue(ciJson, new TypeReference<List<Image>>() {
			});
		} catch (Exception e) {
			e.printStackTrace();
			return Collections.emptyList();
		}
	}

}
