package net.bluemind.core.backup.continuous.events;

import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.hash.Hashing;
import com.google.common.util.concurrent.RateLimiter;

import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.tx.outbox.api.ITxOutbox;

public interface ContinuousContenairization<T> {

	static final Logger logger = LoggerFactory.getLogger(ContinuousContenairization.class);

	static final Map<String, RateLimiter> byTypeLimit = new ConcurrentHashMap<>();

	String type();

	default ITxOutbox targetOutbox(String domainUid, String ownerUid) {
		return TxOutboxLookup.forContainer(descriptor(domainUid, ownerUid));
	}

	default void save(String domainUid, String ownerUid, String itemUid, T value, boolean created) {
		ItemValue<T> iv = itemValue(itemUid, value, created);
		save(domainUid, ownerUid, iv.item(), value);
	}

	default void save(String domainUid, String ownerUid, Item item, T value) {
		ITxOutbox store = targetOutbox(domainUid, ownerUid);
		if (!store.isPaused()) {
			ItemValue<T> iv = ItemValue.create(item, value);
			store.forKafka(iv, null, false);
		}
	}

	default void delete(String domainUid, String ownerUid, String itemUid, T previous) {
		ITxOutbox store = targetOutbox(domainUid, ownerUid);
		if (!store.isPaused()) {
			ItemValue<T> iv = itemValue(itemUid, previous, false);
			store.forKafka(iv, null, true);
		}
	}

	default void log(String operation, ContainerDescriptor metaDesc, ItemValue<T> iv, Throwable ex) {
		String t = type();
		RateLimiter lim = byTypeLimit.computeIfAbsent(t, k -> RateLimiter.create(1.0 / 2));
		if (ex != null) {
			logger.error("{}:fails type:{} domainUid:{} ownerUid:{} itemUid:{}", operation, t, metaDesc.domainUid,
					metaDesc.owner, iv.uid, ex);
		} else if (lim.tryAcquire()) {
			logger.debug("{}:succeed type:{} domainUid:{} ownerUid:{} itemUid:{}", operation, t, metaDesc.domainUid,
					metaDesc.owner, iv.uid);
		}
	}

	default BaseContainerDescriptor descriptor(String domainUid, String ownerUid) {
		return BaseContainerDescriptor.create(ownerUid + "_at_" + domainUid + "_" + type(), ownerUid + " " + type(),
				ownerUid, type(), domainUid, true);
	}

	default ItemValue<T> itemValue(String uid, T identity, boolean created) {
		ItemValue<T> iv = ItemValue.create(uid, identity);
		iv.internalId = Hashing.sipHash24().hashBytes(iv.uid.getBytes()).asLong();
		if (created) {
			iv.created = new Date();
		} else {
			iv.updated = new Date();
		}
		return iv;
	}

}
