/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.ui.gwtsharing.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.HashSet;
import java.util.stream.Collectors;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;

public class AclComparator {

	private final String containerUid;
	private final List<AccessControlEntry> source;
	private final List<AccessControlEntry> target;

	private Set<AccessControlEntry> finalTarget;

	public AclComparator(String containerUid, List<AccessControlEntry> source, List<AccessControlEntry> target) {
		this.containerUid = containerUid;
		this.source = source;
		this.target = target;
		this.finalTarget = new HashSet<>();
	}

	public List<AccessControlEntry> getSource() {
		return this.source;
	}

	public Set<AccessControlEntry> getNewTarget() {
		return this.finalTarget;
	}

	private static List<Verb> sanitizeVerbs(List<Verb> handled, List<Verb> unhandled) {
		Set<Verb> allVerbs = new HashSet<>();
		handled.stream().map(v -> Verb.expand(v)).forEach(v -> allVerbs.addAll(v));
		allVerbs.addAll(unhandled);
		return removeBrokenVerbs(unhandled, allVerbs);
	}

	private static List<Verb> removeBrokenVerbs(List<Verb> unhandled, Set<Verb> dependencies) {
		List<Verb> unsatisfied = unhandled.stream().filter(verb -> !checkRequirements(verb, dependencies)).collect(Collectors.toList());
		dependencies.removeAll(unsatisfied);
		return new ArrayList<>(dependencies);
	}

	private static boolean checkRequirements(Verb verb, Set<Verb> grantedVerbs) {
		Set<Verb> requirements = Verb.expand(verb);
		return requirements.stream().allMatch(v -> grantedVerbs.contains(v));
	}

	private void sanitizeAclVerbs() {
		Map<String, List<AccessControlEntry>> verbsByContainer = target.stream().collect(Collectors.groupingBy(AccessControlEntry::getSubject));

		for (Map.Entry<String, List<AccessControlEntry>> e : verbsByContainer.entrySet()) {
			Map<Boolean, List<Verb>> collect = e.getValue().stream().map(ace -> ace.verb)
					.collect(Collectors.partitioningBy(v -> isHandledVerb(v)));
			List<AccessControlEntry> sanitizedVerbs = sanitizeVerbs(collect.get(true), collect.get(false)).stream()
					.map(v -> AccessControlEntry.create(e.getKey(), v)).collect(Collectors.toList());			
			this.finalTarget.addAll(sanitizedVerbs);
		}
	}

	public boolean aclEquals() {
		sanitizeAclTarget();
		sanitizeAclVerbs();
		if (source.size() == target.size()) {
			for (int i = 0; i < source.size(); i++) {
				if (!aceEquals(source.get(i), target.get(i))) {
					return false;
				}
			}
		} else {
			return false;
		}

		return true;
	}

	private void sanitizeAclTarget() {
		List<AccessControlEntry> unHandledAcl = source.stream().filter(acl -> !isHandledVerb(acl.verb))
				.collect(Collectors.toList());
		
		if (!unHandledAcl.isEmpty()) {
			List<AccessControlEntry> sameAclSubjects = target.stream()
					.filter(acl -> unHandledAcl.stream().anyMatch(a -> a.subject.equals(acl.subject)))
					.collect(Collectors.toList());
			if (!sameAclSubjects.isEmpty() && sameAclSubjects.stream().noneMatch(acl -> !isHandledVerb(acl.verb))) {
				unHandledAcl.stream().forEach(target::add);
			}
		}

		Map<String, List<AccessControlEntry>> result = target.stream()
				.collect(Collectors.groupingBy(AccessControlEntry::getSubject));
		result.entrySet().forEach(r -> {
			if (r.getValue().stream().noneMatch(v -> isHandledVerb(v.verb))) {
				target.removeAll(r.getValue());
			}
		});
	}

	public static boolean isHandledVerb(Verb verb) {
		return verb == Verb.Freebusy || verb == Verb.Invitation || verb == Verb.Read || verb == Verb.Write || verb == Verb.Manage || verb == Verb.All || verb == Verb.Visible;
	}

	private static boolean aceEquals(AccessControlEntry sourceEntry, AccessControlEntry targetEntry) {
		return sourceEntry.subject.equals(targetEntry.subject) && sourceEntry.verb == targetEntry.verb;
	}
}
