package net.bluemind.core.container.hooks.aclchangednotification;

import java.util.ArrayList;
import java.util.List;

import net.bluemind.core.container.hooks.IAclHook;
import net.bluemind.core.container.hooks.aclchangednotification.AclDiff.AclStatus;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.LocalJsonObject;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;

public class AclChangedNotificationHook implements IAclHook {

	@Override
	public void onAclChanged(BmContext context, ContainerDescriptor container, List<AccessControlEntry> previous,
			List<AccessControlEntry> current) {
		if (StateContext.getState() == SystemState.CORE_STATE_CLONING) {
			return;
		}

		previous = AccessControlEntry.compact(previous);
		current = AccessControlEntry.compact(current);

		List<AclDiff> diff = prepareAclDiff(context.getSecurityContext().getSubject(), previous, current);

		if (!diff.isEmpty()) {
			AclChangedMsg aclChangeMsg = new AclChangedMsg(context.getSecurityContext().getSubject(),
					context.getSecurityContext().getContainerUid(), container.uid, container.name, container.type,
					container.ownerDisplayname, diff,
					context.getSecurityContext().getSubject().equals(container.owner));
			VertxPlatform.eventBus().publish(
					AclChangedNotificationVerticle.ACL_CHANGED_NOTIFICATION_COLLECT_BUS_ADDRESS,
					new LocalJsonObject<>(aclChangeMsg));
		}
	}

	private static List<AclDiff> prepareAclDiff(String subject, List<AccessControlEntry> previous,
			List<AccessControlEntry> current) {
	
		List<AccessControlEntry> newAcls = filteredAcls(subject, current, previous);
		List<AccessControlEntry> oldAlcs = filteredAcls(subject, previous, current);
	
		List<AclDiff> diffAcls = new ArrayList<>();
		for (AccessControlEntry oldAce : oldAlcs) {
			// search same subject in new
			addToListForUpdate(diffAcls, newAcls, oldAce);
		}
	
		for (AccessControlEntry newAce : newAcls) {
			// search same subject in old
			addToListForUpdate(diffAcls, oldAlcs, newAce);
		}
	
		if (newAcls.isEmpty()) {
			addToList(diffAcls, oldAlcs, AclStatus.REMOVED);
		} else {
			addToListNoneMatch(diffAcls, oldAlcs, newAcls, AclStatus.REMOVED);
		}
	
		if (oldAlcs.isEmpty()) {
			addToList(diffAcls, newAcls, AclStatus.ADDED);
		} else {
			addToListNoneMatch(diffAcls, newAcls, oldAlcs, AclStatus.ADDED);
		}
	
		return diffAcls;
	}

	private static List<AccessControlEntry> filteredAcls(String subject, List<AccessControlEntry> sourceList,
			List<AccessControlEntry> listToCompare) {
		return sourceList.stream() //
				.filter(e -> e.verb != Verb.Visible && !listToCompare.contains(e) && !e.subject.equals(subject)) //
				.toList();
	}

	private static void addToListForUpdate(List<AclDiff> result, List<AccessControlEntry> sourceList, AccessControlEntry entry) {
		sourceList.stream().filter(v -> v.subject.equals(entry.subject)).findFirst()
				.ifPresent(v -> result.add(AclDiff.createAclDiffForUpdate(v, entry)));
	}

	private static void addToList(List<AclDiff> result, List<AccessControlEntry> sourceList, AclStatus status) {
		sourceList.forEach(aws -> result.add(AclDiff.createAclDiff(aws, status)));
	}

	private static void addToListNoneMatch(List<AclDiff> result, List<AccessControlEntry> sourceList,
			List<AccessControlEntry> listToCompare, AclStatus status) {
		sourceList.stream().filter(s -> listToCompare.stream().noneMatch(v -> v.subject.equals(s.subject)))
				.forEach(aws -> result.add(AclDiff.createAclDiff(aws, status)));
	}
}
