/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.milter.impl;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.netflix.spectator.api.Registry;
import com.sendmail.jilter.JilterProcessor;

import io.netty.buffer.Unpooled;
import io.vertx.core.buffer.Buffer;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;

public class MilterSession {
	private static final Logger logger = LoggerFactory.getLogger(MilterSession.class);
	private static final Registry registry = MetricsRegistry.get();
	private static final IdFactory idFactory = new IdFactory(MetricsRegistry.get(), MilterHandler.class);
	private static final int BUFFER_SIZE = 1024;

	private final SocketChannel channel;
	private final JilterProcessor jilterProcessor;
	private Selector selector;
	private final long start;

	private final Queue<ByteBuffer> writeQueue = new ConcurrentLinkedQueue<>();
	private ByteBuffer responseBuffer;
	private static final boolean RECORD_TRAFFIC = new File("/etc/bm", "milter.record").exists();
	private final ITrafficRecorder rec;

	public MilterSession(SocketChannel channel, MilterHandler handler, Selector selector, String remoteAddress) {
		this.channel = channel;
		this.jilterProcessor = new JilterProcessor(handler);
		this.selector = selector;
		this.start = registry.clock().monotonicTime();
		if (RECORD_TRAFFIC) {
			rec = new TrafficRecorder(remoteAddress);
		} else {
			rec = buf -> {
			};
		}
		registry.counter(idFactory.name("connectionsCount")).increment();
	}

	public ByteBuffer read() throws IOException {
		ByteBuffer commandBuffer = ByteBuffer.allocate(BUFFER_SIZE);
		int readBytes = channel.read(commandBuffer);
		if (readBytes <= 0) {
			return null;
		}
		rec.record(Buffer.buffer(Unpooled.wrappedBuffer(commandBuffer)));

		return commandBuffer;
	}

	public void processCommands(ByteBuffer commandBuffer) {
		commandBuffer.flip(); // Switch to returnad mode
		processCommand(commandBuffer);
	}

	private void processCommand(ByteBuffer commandBuffer) {
		try {
			if (processCommandData(commandBuffer)) {
				queueResponse();
			}
		} catch (Exception e) {
			logger.warn("Error processing command: ", e);
		}
	}

	private boolean processCommandData(ByteBuffer commandBuffer) throws IOException {

		WritableByteChannel responseChannel = new WritableByteChannel() {

			@Override
			public boolean isOpen() {
				return true;
			}

			@Override
			public void close() throws IOException {
				closeSession();
			}

			@Override
			public int write(ByteBuffer src) throws IOException {
				int totalBytesSent = 0;

				// Make sure buffer is in read mode
				if (src.position() > 0 && src.limit() == src.capacity()) {
					logger.debug("Buffer is not in read mode, call flip()");
					src.flip();
				}

				// Check whether the buffer contains data
				if (!src.hasRemaining()) {
					logger.debug("Empty buffer, nothing to send");
					return 0;
				}

				// Write loop to ensure everything is sent
				while (src.hasRemaining()) {
					responseBuffer = ByteBuffer.allocate(src.remaining());
					responseBuffer.put(src);
					responseBuffer.flip();
					writeQueue.offer(responseBuffer);

					totalBytesSent += responseBuffer.remaining();
				}

				logger.debug("Wrote response {} bytes", totalBytesSent);
				return totalBytesSent;
			}
		};

		return jilterProcessor.process(responseChannel, commandBuffer);
	}

	private void queueResponse() {
		// Request write notification from selector
		try {
			SelectionKey key = channel.keyFor(selector);
			if (key != null && key.isValid()) {
				key.interestOps(key.interestOps() | SelectionKey.OP_WRITE);
				selector.wakeup();
			}
		} catch (Exception e) {
			logger.warn("Error requesting write notification", e);
		}
	}

	public boolean write() throws IOException {
		while (!writeQueue.isEmpty()) {
			ByteBuffer buffer = writeQueue.peek();
			if (buffer == null)
				break;

			int bytesWritten = channel.write(buffer);
			logger.debug("Wrote {} bytes to {}", bytesWritten, channel.getRemoteAddress());

			if (buffer.hasRemaining()) {
				return false; // Channel buffer full, try again later
			}

			writeQueue.poll(); // Remove completed buffer
		}

		return true; // All data written
	}

	public void closeSession() {
		registry.timer(idFactory.name("sessionDuration")).record(registry.clock().monotonicTime() - start,
				TimeUnit.NANOSECONDS);
		try {
			channel.close();
		} catch (IOException e) {
			logger.error("Error closing channel", e);
		}
	}

}
