/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.milter.impl;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Java Milter Server using NIO SocketChannel and Selector with Virtual Threads
 * for message processing
 */
public class MilterMainServer {
	private static final Logger logger = LoggerFactory.getLogger(MilterMainServer.class);

	private final int port;
	private final String bindAddress;
	private final ConcurrentHashMap<SocketChannel, Selector> clientSelectors = new ConcurrentHashMap<>();
	private ServerSocketChannel serverChannel;
	private Selector acceptorSelector;
	private AtomicBoolean running = new AtomicBoolean();
	private final ExecutorService processingExecutor;

	public MilterMainServer(int port, String bindAddress) {
		this.port = port;
		this.bindAddress = bindAddress;
		this.processingExecutor = Executors.newVirtualThreadPerTaskExecutor();
	}

	public void start() throws IOException {
		acceptorSelector = Selector.open();
		serverChannel = ServerSocketChannel.open();
		serverChannel.configureBlocking(false);
		serverChannel.bind(new InetSocketAddress(bindAddress, port));
		serverChannel.register(acceptorSelector, SelectionKey.OP_ACCEPT);

		running.set(true);
		logger.info("NIO Milter server started on {}:{}", bindAddress, port);

		// Main selector loop in a virtual thread
		Thread.startVirtualThread(this::selectorLoop);
	}

	public void stop() throws IOException {
		if (running.compareAndSet(true, false)) {

			// Close all client selectors
			for (Selector selector : clientSelectors.values()) {
				selector.wakeup();
			}

			processingExecutor.shutdown();

			if (acceptorSelector != null) {
				acceptorSelector.wakeup();
			}

			if (serverChannel != null) {
				serverChannel.close();
			}
			if (acceptorSelector != null) {
				acceptorSelector.close();
			}

		}
		logger.info("NIO Milter server stopped");
	}

	private void selectorLoop() {
		try {
			while (running.get()) {
				if (acceptorSelector.select() > 0) {
					Set<SelectionKey> selectedKeys = acceptorSelector.selectedKeys();
					Iterator<SelectionKey> iterator = selectedKeys.iterator();

					while (iterator.hasNext()) {
						SelectionKey key = iterator.next();
						iterator.remove();

						if (key.isAcceptable()) {
							acceptConnection();
						}
					}
				}
			}
		} catch (IOException e) {
			logger.error("Selector loop error", e);
		}
	}

	private void acceptConnection() throws IOException {
		SocketChannel clientChannel = serverChannel.accept();
		if (clientChannel != null) {
			clientChannel.configureBlocking(false);
			logger.info("Accepted connection from: {}", clientChannel.getRemoteAddress());

			processingExecutor.submit(new MilterConnection(clientChannel, clientSelectors, running));
		}
	}

}