/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.text.Normalizer;
import java.util.regex.Pattern;

import com.datastax.oss.driver.shaded.guava.common.base.CharMatcher;

public class SubjectCleaner {

	private static final Pattern STRIP_RE = Pattern.compile("^re\s*:", Pattern.CASE_INSENSITIVE);
	private static final CharMatcher NOT_ALPHANUM = CharMatcher.inRange('a', 'z').or(CharMatcher.inRange('0', '9'))
			.or(CharMatcher.inRange('A', 'Z')).negate().precomputed();

	private SubjectCleaner() {
	}

	public static String clean(String s) {
		if (s == null) {
			return "";
		}
		String subject = unaccent(s);
		String leadingNonAsciiStripped = NOT_ALPHANUM.trimLeadingFrom(subject);

		String noRe = STRIP_RE.matcher(leadingNonAsciiStripped).replaceFirst("");
		return CharMatcher.whitespace().trimLeadingFrom(noRe);
	}

	public static String cleanKeepPrefix(String s) {
		String un = unaccent(s);
		return CharMatcher.ascii().negate().removeFrom(un);
	}

	private static String unaccent(String src) {
		if (src == null) {
			return "";
		}
		return Normalizer.normalize(src, Normalizer.Form.NFD).replaceAll("[^\\p{ASCII}]", "");
	}
}
