/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.sql.SQLException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BoundStatement;
import com.datastax.oss.driver.api.core.cql.Row;
import com.datastax.oss.driver.api.core.uuid.Uuids;

import net.bluemind.backend.mail.replica.api.MailboxRecordExpunged;
import net.bluemind.backend.mail.repository.IMailboxRecordExpungedStore;

public class CqlMailboxRecordExpungedStore extends BaseExpungedStore implements IMailboxRecordExpungedStore {

	public CqlMailboxRecordExpungedStore(CqlSession s) {
		super(s);
	}

	@Override
	public List<MailboxRecordExpunged> getExpiredItems(int days) throws SQLException {
		Instant targetDate = Instant.now().minus(days, ChronoUnit.DAYS);
		return getExpiredItemBefore(targetDate);
	}

	public List<MailboxRecordExpunged> getExpiredItemBefore(Instant targetDate) {
		String q = """
				SELECT toTimestamp(created), imap_uid, subtree_id, container_id, item_id
				FROM q_mailbox_record_expunged
				where global_id=20120331 and created < ?
				limit 5000
				""";
		BoundStatement bound = session.prepare(q).bind(Uuids.startOf(targetDate.toEpochMilli()));
		var res = session.execute(bound);
		List<MailboxRecordExpunged> ret = new ArrayList<>(5000);
		Iterator<Row> it = res.iterator();
		while (it.hasNext() && ret.size() < 5000) {
			Row r = it.next();
			Instant date = r.getInstant(0);
			if (date.isBefore(targetDate)) {
				MailboxRecordExpunged re = new MailboxRecordExpunged();
				re.created = Date.from(date);
				re.imapUid = r.getLong(1);
				re.subtreeId = r.getLong(2);
				re.containerId = r.getLong(3);
				re.itemId = r.getLong(4);
				ret.add(re);
			} else {
				break;
			}
		}
		return ret;
	}

	@Override
	public void deleteExpunged(long sub, long rec, long itemId) throws SQLException {
		baseDelete(sub, rec, itemId);
	}

}
