/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.util.Date;
import java.util.List;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.backend.mail.replica.api.MailboxRecordExpunged;
import net.bluemind.backend.mail.repository.IMailboxRecordExpungedByFolderStore;
import net.bluemind.core.container.model.Container;

public class CqlMailboxRecordExpungedByFolderStore extends BaseExpungedStore
		implements IMailboxRecordExpungedByFolderStore {

	private Container subtree;
	private Container mbox;

	public CqlMailboxRecordExpungedByFolderStore(CqlSession s, Container subtree, Container mbox) {
		super(s);
		this.subtree = subtree;
		this.mbox = mbox;
	}

	@Override
	public void delete(Long itemId) {
		baseDelete(subtree.id, mbox.id, itemId);
	}

	@Override
	public MailboxRecordExpunged get(Long itemId) {

		String q = """
				SELECT toTimestamp(created), imap_uid
				FROM q_mailbox_record_expunged_by_container
				WHERE subtree_id=? AND container_id=? AND item_id=?
				""";
		return unique(q, r -> {
			MailboxRecordExpunged e = new MailboxRecordExpunged();
			e.containerId = mbox.id;
			e.subtreeId = subtree.id;
			return e;
		}, (r, i, e) -> {
			e.created = Date.from(r.getInstant(i++));
			e.imapUid = r.getLong(i++);
			e.itemId = itemId;
			return i;
		}, subtree.id, mbox.id, itemId);
	}

	@Override
	public List<MailboxRecordExpunged> fetch() {
		String q = """
				SELECT toTimestamp(created), imap_uid, item_id
				FROM q_mailbox_record_expunged_by_container
				WHERE subtree_id=? AND container_id=?
				""";
		return map(q, r -> {
			MailboxRecordExpunged e = new MailboxRecordExpunged();
			e.containerId = mbox.id;
			e.subtreeId = subtree.id;
			return e;
		}, (r, i, e) -> {
			e.created = Date.from(r.getInstant(i++));
			e.imapUid = r.getLong(i++);
			e.itemId = r.getLong(i++);
			return i;
		}, subtree.id, mbox.id);
	}

	@Override
	public Long count() {
		return (long) fetch().size();
	}

}
