/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.sql.SQLException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.Row;

import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.MessageBody.Header;
import net.bluemind.backend.mail.api.MessageBody.Part;
import net.bluemind.backend.mail.api.MessageBody.Recipient;
import net.bluemind.backend.mail.repository.IMessageBodyStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ListReader;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.cql.persistence.CqlAbstractStore;

public class CqlBodyStore extends CqlAbstractStore implements IMessageBodyStore {

	private static final Logger logger = LoggerFactory.getLogger(CqlBodyStore.class);

	private static final ListReader<Header> headersReader = JsonUtils.listReader(Header.class);
	private static final ListReader<Recipient> recipientReader = JsonUtils.listReader(Recipient.class);
	private static final ValueReader<Part> partReader = JsonUtils.reader(Part.class);

	public CqlBodyStore(CqlSession s) {
		super(s);
	}

	private static final EntityPopulator<MessageBody> POP = (Row r, int i, MessageBody mb) -> {
		mb.guid = r.getString(i++);
		mb.subject = r.getString(i++);
		mb.structure = partReader.read(r.getString(i++));
		mb.headers = headersReader.read(r.getString(i++));
		mb.recipients = recipientReader.read(r.getString(i++));
		mb.messageId = r.getString(i++);
		mb.references = r.getList(i++, String.class);
		mb.date = dateOrNull(r, i++);
		mb.size = r.getInt(i++);
		mb.preview = r.getString(i++);
		mb.created = Date.from(r.getInstant(i++));
		mb.bodyVersion = r.getByte(i++);
		return i;
	};

	@Override
	public int store(MessageBody value) {
		if (value.created == null) {
			value.created = new Date();
		}
		boolean applied = voidCql("""
				INSERT INTO t_message_body
				(guid, subject, structure, headers, recipients,
				message_id, references_header,
				date_header, size, preview, body_version, created)
				VALUES
				(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
				IF NOT EXISTS
				""", value.guid, value.subject, JsonUtils.asString(value.structure), JsonUtils.asString(value.headers),
				JsonUtils.asString(value.recipients), value.messageId, value.references,
				value.date != null ? value.date.toInstant() : null, value.size,
				SubjectCleaner.cleanKeepPrefix(value.preview), (byte) value.bodyVersion, value.created.toInstant());
		if (applied) {
			markOrphan(value.guid);
			return 1;
		} else {
			logger.debug("Skip existing body {}", value.guid);
			return 0;
		}

	}

	@Override
	public MessageBody get(String guid) {
		return unique("""
				SELECT
				guid, subject, structure, headers, recipients,
				message_id, references_header,
				date_header, size, preview, created, body_version
				FROM t_message_body
				WHERE guid = ?
				""", r -> new MessageBody(), POP, guid);
	}

	@Override
	public List<MessageBody> multiple(List<String> guids) throws SQLException {
		return map("""
				SELECT
				guid, subject, structure, headers, recipients,
				message_id, references_header,
				date_header, size, preview, created, body_version
				FROM t_message_body
				WHERE guid IN ?
				""", r -> new MessageBody(), POP, guids);
	}

	@Override
	public boolean exists(String guid) {
		var prep = session.prepare("""
				SELECT guid FROM t_message_body WHERE guid = ?
				""");
		return session.execute(prep.bind(guid)).one() != null;
	}

	@Override
	public List<String> existing(List<String> toCheck) throws SQLException {
		return toCheck.stream().map(guid -> exists(guid) ? guid : null).filter(Objects::nonNull).toList();
	}

	@Override
	public List<String> deleteOrphanBodies() throws SQLException {

		Instant randomBarrier = Instant.now().minus(2, ChronoUnit.DAYS);
		List<String> bodies = map("""
				SELECT guid FROM q_message_body_orphaned
				WHERE orphaned_at < ?
				LIMIT 500 ALLOW FILTERING
				""", r -> r.getString(0), voidPop(), randomBarrier);
		if (bodies.isEmpty()) {
			return Collections.emptyList();
		}
		var res = batch(//
				b("DELETE FROM t_message_body WHERE guid IN ?", bodies),
				b("UPDATE q_message_body_orphaned SET pruned_at=toUnixTimestamp(now()) WHERE guid IN ?", bodies)//
		);
		if (res.wasApplied()) {
			logger.info("Pruned {} orphaned bodies", bodies.size());
			return bodies;
		} else {
			return Collections.emptyList();
		}
	}

	@Override
	public List<String> deletePurgedBodies(Instant removedBefore, long limit) throws SQLException {
		// TODO Auto-generated method stub
		return Collections.emptyList();
	}

	public void refBody(String bodyGuid, Container mbox, Item rec) {
		voidCql("insert into t_body_references (guid, container_id, item_id) values (?,?,?)", bodyGuid, mbox.id,
				rec.id);
		voidCql("delete from q_message_body_orphaned where guid=?", bodyGuid);
	}

	public void unrefBody(String bodyGuid, Container mbox, Item rec) {
		voidCql("delete from t_body_references where guid=? and container_id=? and item_id=?", bodyGuid, mbox.id,
				rec.id);
		checkIfOrphan(bodyGuid);
	}

	private void checkIfOrphan(String bodyGuid) {
		List<Long> otherRefs = map("""
				SELECT item_id FROM t_body_references WHERE guid=?
				""", r -> r.getLong(0), voidPop(), bodyGuid);
		if (otherRefs.isEmpty()) {
			markOrphan(bodyGuid);
			logger.info("body {} is orphaned", bodyGuid);
		}

	}

	private void markOrphan(String bodyGuid) {
		applyCql("INSERT INTO q_message_body_orphaned (guid, orphaned_at) VALUES (?, toUnixTimestamp(now()))",
				bodyGuid);
	}

	public List<String> lowerThan(int version, List<String> guids) {
		return map("""
				SELECT
				guid
				FROM t_message_body
				WHERE guid IN ? AND body_version < ?
				ALLOW FILTERING
				""", r -> r.getString(0), voidPop(), guids, (byte) version);
	}

	@Override
	public void deleteNoRecordNoPurgeOrphanBodies() throws SQLException {
		// TODO Auto-generated method stub
	}

}
