/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.addressbook.impl.pg;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.addressbook.api.IVCardService;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.sessions.Sessions;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.email.SendVcf;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;

public class SendUserBookVCFTaskPg {
	private static final Logger logger = LoggerFactory.getLogger(SendUserBookVCFTaskPg.class);
	private IMonitoredRestoreRestorableItem restorableItem;
	private DataProtectGeneration backup;
	private ResourceBundle bundle;
	private final RestoreActionExecutor<EmailData> executor;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@SuppressWarnings("unchecked")
	@Deprecated
	public SendUserBookVCFTaskPg(DataProtectGeneration backup, ResourceBundle bundle,
			IMonitoredRestoreRestorableItem restorableItem,
			RestoreActionExecutor<? extends IRestoreActionData> executor) {
		this.backup = backup;
		this.bundle = bundle;
		this.restorableItem = restorableItem;
		this.executor = (RestoreActionExecutor<EmailData>) executor;
	}

	public void run() {
		SendVcf sendEmail = new SendVcf(restorableItem, executor);

		SecurityContext backUserContext = as(restorableItem.entryUid(), restorableItem.domain());
		try (BackupDataProvider bdp = new BackupDataProvider(null, backUserContext, restorableItem.getMonitor())) {
			IServiceProvider back = bdp.createContextWithData(backup, restorableItem.item()).provider();

			IContainers containersService = back.instance(IContainers.class);
			ContainerQuery cq = ContainerQuery.ownerAndType(backUserContext.getSubject(), IAddressBookUids.TYPE);
			List<ContainerDescriptor> books = containersService.all(cq);

			Map<String, String> allVCards = HashMap.newHashMap(books.size());
			for (ContainerDescriptor b : books) {
				IVCardService service = back.instance(IVCardService.class, b.uid);
				allVCards.put(b.name, service.exportAll());
			}

			restorableItem.monitorLog("Export '{}' contacts from {} [uid={}]", allVCards.size(),
					restorableItem.displayName(), restorableItem.entryUid());

			sendEmail.sendMessage(allVCards, bundle.getString("send.addressBook.restore.message"),
					bundle.getString("send.addressBook.restore.subject"));
		} catch (Exception e) {
			logger.error("Error while sending user addressbooks", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.ADDRESSBOOK));
		} finally {
			restorableItem.endTask();
		}
	}

	private static final SecurityContext as(String uid, String domainContainerUid) throws ServerFault {
		SecurityContext userContext = new SecurityContext(UUID.randomUUID().toString(), uid, Arrays.<String>asList(),
				Arrays.<String>asList(), Collections.emptyMap(), domainContainerUid, "en", "RestoreUserBooksTask.as");
		Sessions.get().put(userContext.getSessionId(), userContext);
		return userContext;
	}

}
