/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.addressbook.impl.pg;

import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.AddressBookDescriptor;
import net.bluemind.addressbook.api.IAddressBook;
import net.bluemind.addressbook.api.IAddressBooksMgmt;
import net.bluemind.addressbook.api.IVCardService;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.email.SendVcf;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;

public class SendDomainBookVCFTaskPg {
	private static final Logger logger = LoggerFactory.getLogger(SendDomainBookVCFTaskPg.class);
	private IMonitoredRestoreRestorableItem restorableItem;
	private DataProtectGeneration backup;
	private ResourceBundle bundle;
	private final RestoreActionExecutor<EmailData> executor;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@SuppressWarnings("unchecked")
	@Deprecated
	public SendDomainBookVCFTaskPg(DataProtectGeneration backup, ResourceBundle bundle,
			IMonitoredRestoreRestorableItem restorableItem,
			RestoreActionExecutor<? extends IRestoreActionData> executor) {
		this.backup = backup;
		this.bundle = bundle;
		this.restorableItem = restorableItem;
		this.executor = (RestoreActionExecutor<EmailData>) executor;
	}

	public void run() {
		SendVcf sendEmail = new SendVcf(restorableItem, executor);
		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext back = bdp.createContextWithData(backup, restorableItem.item());
			AddressBookDescriptor backDomainAddressBook = back.provider().instance(IAddressBooksMgmt.class)
					.get(restorableItem.item().entryUid);
			List<String> vCardUids = back.provider().instance(IAddressBook.class, restorableItem.item().entryUid)
					.allUids();

			restorableItem.monitorLog("Export '{}' contacts from {} [uid={}]", vCardUids.size(),
					backDomainAddressBook.name, restorableItem.entryUid());

			IVCardService service = back.provider().instance(IVCardService.class, restorableItem.item().entryUid);
			Map<String, String> allVCards = Map.of(backDomainAddressBook.name, service.exportCards(vCardUids));
			sendEmail.sendMessage(allVCards, bundle.getString("send.addressBook.restore.message"),
					bundle.getString("send.addressBook.restore.subject"));
		} catch (Exception e) {
			logger.error("Error while sending domain addressbooks", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.DOMAIN_ADDRESSBOOK));
		} finally {
			restorableItem.endTask();
		}
	}
}
