/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  * 
  * 
  */
/*
 * MIT License
	Copyright (c) 2017 Eugen Paraschiv
	
	Permission is hereby granted, free of charge, to any person obtaining a copy
	of this software and associated documentation files (the "Software"), to deal
	in the Software without restriction, including without limitation the rights
	to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
	copies of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions:
	
	The above copyright notice and this permission notice shall be included in all
	copies or substantial portions of the Software.
	
	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
	SOFTWARE.
 */
package net.bluemind.core.backup.continuous.tools;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;

public class LockByKey<T> {
	private static final Logger logger = LoggerFactory.getLogger(LockByKey.class);

	private static class LockWrapper {
		private final ReentrantLock lock = new ReentrantLock();
		private int numberOfThreadsInQueue = 1;

		private LockWrapper addThreadInQueue() {
			numberOfThreadsInQueue++;
			return this;
		}

		private int removeThreadFromQueue() {
			return --numberOfThreadsInQueue;
		}
	}

	private final ConcurrentHashMap<T, LockWrapper> locks = new ConcurrentHashMap<>();

	public void lock(T key) {
		Stopwatch chrono = Stopwatch.createStarted();
		LockWrapper lockWrapper = locks.compute(key, (k, v) -> v == null ? new LockWrapper() : v.addThreadInQueue());
		lockWrapper.lock.lock();
		long grab = chrono.elapsed(TimeUnit.MILLISECONDS);
		if (grab > 100) {
			logger.warn("Locking on key {} was slow ({}ms)", key, grab);
		}
	}

	public void unlock(T key) {
		locks.compute(key, (k, lockWrapper) -> {
			if (lockWrapper != null) {
				lockWrapper.lock.unlock();
				return lockWrapper.removeThreadFromQueue() == 0 ? null : lockWrapper;
			}
			return null;
		});
	}

}