/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.backup.continuous.restore.queues;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.util.concurrent.RateLimiter;

import net.bluemind.backend.mail.replica.api.IInternelMessageBodyPurgeQueue;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.restore.domains.RestoreDomainType;
import net.bluemind.core.backup.continuous.restore.domains.RestoreLogger;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.helper.ArchiveHelper;
import net.bluemind.system.sysconf.helper.SysConfHelper;

public class RestoreMessageBodyPurgeQueue implements RestoreDomainType {
	private RestoreLogger log;

	private final ValueReader<MessageBodyPurgeQueue> reader = JsonUtils
			.reader(new TypeReference<MessageBodyPurgeQueue>() {
			});

	private IInternelMessageBodyPurgeQueue api;
	private static final int DEFAULT_RETENTION_DAYS = 90;

	private final int sdsBackupRententionDays;

	private RateLimiter ignoreLogs;
	private static Logger logger = LoggerFactory.getLogger(RestoreMessageBodyPurgeQueue.class);

	public RestoreMessageBodyPurgeQueue(RestoreLogger log, IServiceProvider target) {
		this.log = log;
		this.api = target.instance(IInternelMessageBodyPurgeQueue.class);
		this.sdsBackupRententionDays = getRetentionDays();
		ignoreLogs = RateLimiter.create(10);
	}

	private int getRetentionDays() {
		SystemConf sysconf = SysConfHelper.fromSharedMap();
		Integer days = sysconf.integerValue(SysConfKeys.sds_backup_rentention_days.name());
		if (ArchiveHelper.isSdsArchiveKind(sysconf)) {
			return days == null ? DEFAULT_RETENTION_DAYS : days;
		} else {
			return 0;
		}
	}

	@Override
	public String type() {
		return "pq";
	}

	@Override
	public void restore(RecordKey key, String payload) {
		if (key.operation.equals("TRUNCATE")) {
			log.delete("MessageBodyPurgeQueue", key);
			api.deleteAll();
			return;
		}

		String guid = key.uid;
		if (payload == null || payload.isEmpty()) {
			log.delete("MessageBodyPurgeQueue", key);
			api.delete(guid);
			return;
		}
		MessageBodyPurgeQueue pq = reader.read(payload);
		if (pq.created() == null) {
			logger.debug("{} payload is not empty, not null, but no created is set {} {}", guid, pq, payload);
			log.delete("MessageBodyPurgeQueue", key);
			api.delete(guid);
		} else {
//			if (pq.created()
//					.isBefore(Instant.now().minusSeconds(Duration.ofDays(sdsBackupRententionDays).toSeconds()))) {
//				if (ignoreLogs.tryAcquire()) {
//					logger.info("Ignoring {} message: created too long ago", guid);
//				}
//			} else {
			log.create("MessageBodyPurgeQueue", key);
			api.createOrUpdate(guid, pq.created(), pq.removed(), pq.immediate_remove());
//			}
		}

	}

}
