/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.backup.continuous.restore.index;

import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import io.vertx.core.json.JsonObject;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.backup.continuous.DataElement;
import net.bluemind.core.backup.continuous.index.IStreamLiveIndex;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.restore.IOwnerChecker;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;

public record Indices(ItemValue<Domain> domain, IDirectory dirApi, Set<String> wellKnownOwners,
		IStreamLiveIndex<String, byte[]> delIndex, IStreamLiveIndex<String, JsonObject> dirIndex)
		implements IOwnerChecker {

	private static final Logger logger = LoggerFactory.getLogger(Indices.class);

	private static final Set<String> brokenDirEntries = loadBrokenEntriesFromConfig();

	@Override
	public boolean isKnown(String owner) {
		if (brokenDirEntries.contains(owner)) {
			return false;
		}
		return wellKnownOwners.contains(owner) || dirIndex.get(owner) != null;
	}

	@Override
	public boolean isKnownDeletion(String owner) {
		return brokenDirEntries.contains(owner) || delIndex.get(IndexKeys.deletion(domain.uid, owner)) != null;
	}

	@Override
	public boolean isKnownDeletion(RecordKey k) {
		return brokenDirEntries.contains(k.owner) || delIndex.get(IndexKeys.deletion(k)) != null;
	}

	private static Set<String> loadBrokenEntriesFromConfig() {
		Config coreConfig = CoreConfig.get();
		return coreConfig.getStringList(CoreConfig.Clone.BROKEN_DIR_ENTRIES).stream().collect(Collectors.toSet());
	}

	public void close() {
		dirIndex.close();
		delIndex.close();
	}

	@Override
	public Optional<DataElement> outOfOrderCreate(String owner) {
		JsonObject asJs = dirIndex.get(owner);
		boolean possible = asJs != null && dirApi.findByEntryUid(owner) == null;
		if (!possible) {
			return Optional.empty();
		}
		if (logger.isInfoEnabled()) {
			logger.info("OUT_OF_ORDER {} -> {}", owner, asJs.encode());
		}
		RecordKey rk = new RecordKey("dir", "system", domain.uid, asJs.getLong("internalId"),
				asJs.getString("valueClass"), "CREATE");
		DataElement de = new DataElement();
		de.key = rk;
		de.payload = asJs.encode().getBytes();
		return Optional.of(de);

	}

}