package net.bluemind.core.backup.continuous.restore.domains.replication;

import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.io.CountingInputStream;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.streams.ReadStream;
import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.parsing.BodyStreamProcessor;
import net.bluemind.backend.mail.parsing.BodyStreamProcessor.MessageBodyData;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.backup.continuous.dto.VersionnedItem;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.model.RecordKey.Operation;
import net.bluemind.core.backup.continuous.restore.domains.RestoreDomainType;
import net.bluemind.core.backup.continuous.restore.domains.RestoreLogger;
import net.bluemind.core.backup.continuous.restore.domains.RestoreState;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.IRestoreItemCrudSupport;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.vertx.BufferReadStream;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.core.rest.vertx.VertxStream.ReadStreamStream;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.delivery.conversationreference.api.IConversationReference;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.server.api.Server;

public class RestoreMailboxRecords implements RestoreDomainType {

	private static final Logger logger = LoggerFactory.getLogger(RestoreMailboxRecords.class);

	private final ValueReader<VersionnedItem<MailboxRecord>> recReader = JsonUtils
			.reader(new TypeReference<VersionnedItem<MailboxRecord>>() {
			});

	private final RestoreState state;
	private final IServiceProvider provider;
	private final ItemValue<Domain> domain;
	private final RestoreLogger log;
	private final IContainers contApi;

	public RestoreMailboxRecords(RestoreLogger log, RestoreState state, ItemValue<Domain> domain,
			IServiceProvider provider) {
		this.state = state;
		this.provider = provider;
		this.domain = domain;
		this.log = log;
		contApi = provider.instance(IContainers.class);
		logger.debug("preparing for records restore in {}", domain);
	}

	public void restore(RecordKey key, String payload) {
		if (!contApi.exists(key.uid)) {
			log.skip(type(), key, payload);
			return;
		}
		IDbMailboxRecords api = api(key);
		if (Operation.isDelete(key)) {
			delete(key, payload, api);
		} else {
			filterCreateOrUpdate(key, payload, api);
		}
	}

	public String type() {
		return IMailReplicaUids.MAILBOX_RECORDS;
	}

	protected ValueReader<VersionnedItem<MailboxRecord>> reader() {
		return recReader;
	}

	private IDbMailboxRecords api(RecordKey key) {
		String uniqueId = IMailReplicaUids.getUniqueId(key.uid);
		return provider.instance(IDbMailboxRecords.class, uniqueId);
	}

	private void filterCreateOrUpdate(RecordKey key, String payload, IDbMailboxRecords api) {
		VersionnedItem<MailboxRecord> item = reader().read(payload);
		createOrUpdate(api, key, item);
	}

	private void createOrUpdate(IDbMailboxRecords api, RecordKey key, VersionnedItem<MailboxRecord> item) {
		boolean exists = exists(api, key, item);

		IDbMessageBodies apiMessageBody = apiMessageBody(key);
		String guid = item.value.messageBody;
		boolean bodyIsPlaceholder = false;
		if (apiMessageBody != null && item.value.messageBody != null && !apiMessageBody.exists(guid)) {
			ItemValue<MessageBody> itemMessageBody = new ItemValue<>();
			MessageBody mb = new MessageBody();
			mb.guid = guid;
			mb.created = item.created;
			mb.date = item.created;
			itemMessageBody.value = mb;
			apiMessageBody.restore(itemMessageBody, false);
			bodyIsPlaceholder = true;
		}

		if (exists) {
			log.update(type(), key);
			update(api, key, item);
		} else {
			// on update, we don't expect the message to change conversation
			restoreConversationReference(key, item, api, apiMessageBody, guid, bodyIsPlaceholder);

			log.create(type(), key);
			create(api, key, item);
		}
	}

	private void restoreConversationReference(RecordKey key, VersionnedItem<MailboxRecord> item,
			IRestoreItemCrudSupport<MailboxRecord> api, IDbMessageBodies apiMessageBody, String guid,
			boolean bodyIsPlaceholder) {
		MessageBody forRefs = null;
		if (bodyIsPlaceholder && api instanceof IDbMailboxRecords dbApi) {
			// parse from s3
			try {
				Stream inMemStream = dbApi.fetchCompleteByGuid(item.value.messageBody);
				ReadStream<Buffer> b = VertxStream.read(inMemStream);
				if (b instanceof ReadStreamStream rss && rss.unwrap() instanceof BufferReadStream brs) {
					ByteBuf mmaped = brs.nettyBuffer();
					MessageBodyData reparsed = BodyStreamProcessor
							.parseBodyGetFullContent(new CountingInputStream(new ByteBufInputStream(mmaped)));
					forRefs = reparsed.body;
				}
			} catch (Exception e) {
				log.skip("conv_ref", key, guid);
			}
		} else if (apiMessageBody != null) {
			forRefs = apiMessageBody.getComplete(guid);
		}
		if (forRefs != null && item.value.conversationId != null && item.value.conversationId != 0L) {
			String ownerUid = key.owner;
			IConversationReference convRef = provider.instance(IConversationReference.class, domain.uid, ownerUid);
			convRef.restore(item.value.conversationId, forRefs.messageId, forRefs.references);
		}
	}

	private boolean exists(IDbMailboxRecords api, RecordKey key, VersionnedItem<MailboxRecord> item) {
		ItemValue<MailboxRecord> previous = api.getComplete(item.uid);
		if (previous != null && (previous.internalId != item.internalId || previous.value == null)) {
			log.deleteByProduct(type(), key);
			delete(api, key, item.uid);
			return false;
		} else {
			return previous != null;
		}
	}

	private String extractUidFromPayload(String payload) throws IOException {
		try (JsonParser parser = new JsonFactory().createParser(payload)) {
			while (parser.nextToken() != null) {
				if ("value".equals(parser.currentName()) && parser.nextToken() == JsonToken.START_OBJECT) {
					while (parser.nextToken() != JsonToken.END_OBJECT) {
						if ("uid".equals(parser.currentName())) {
							parser.nextToken();
							return parser.getText();
						}
					}
				}
			}
		}
		return null;
	}

	private void delete(RecordKey key, String payload, IDbMailboxRecords api) {
		try {
			log.delete(type(), key);
			delete(api, key, extractUidFromPayload(payload));
		} catch (ServerFault sf) {
			if (!ErrorCode.NOT_FOUND.equals(sf.getCode())) {
				throw sf;
			}
		} catch (IOException e) {
			throw new ServerFault("extract of uid from value in payload " + payload + " failed", e);
		}
	}

	private void delete(IDbMailboxRecords api, RecordKey key, String uid) {
		api.delete(uid);
		api.deleteById(key.id);
	}

	private void create(IRestoreItemCrudSupport<MailboxRecord> api, RecordKey key, VersionnedItem<MailboxRecord> item) {
		ItemValue<MailboxRecord> toRestore = map(item, true);
		try {
			api.restore(toRestore, true);
		} catch (ServerFault sf) {
			if (sf.getCode() == ErrorCode.ALREADY_EXISTS) {
				log.failureIgnored(type(), key, "Item already exists and can't be created, trying to update.");
				api.restore(item, false);
			}
		}
	}

	private ItemValue<MailboxRecord> map(VersionnedItem<MailboxRecord> item, boolean isCreate) {
		return item;
	}

	private void update(IRestoreItemCrudSupport<MailboxRecord> api, RecordKey key, VersionnedItem<MailboxRecord> item) {
		ItemValue<MailboxRecord> toRestore = map(item, false);
		api.restore(toRestore, false);
	}

	private IDbMessageBodies apiMessageBody(RecordKey key) {
		String ownerUid = key.owner;
		ItemValue<Mailbox> mbox = state.getMailbox(ownerUid);
		if (mbox == null) {
			log.skip(type(), key, null);
			return null;
		}
		ItemValue<Server> imap = state.getServer(mbox.value.dataLocation);
		CyrusPartition partition = CyrusPartition.forServerAndDomain(imap, domain.uid);
		return provider.instance(IDbMessageBodies.class, partition.name);
	}

}
