/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2021
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.backup.continuous.restore.domains;

import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;

import io.vertx.core.json.JsonObject;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.model.RecordKey.Operation;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.domain.api.Domain;
import net.bluemind.exchange.mapi.api.IMapiFolderAssociatedInformation;
import net.bluemind.exchange.mapi.api.MapiFAI;

public class RestoreMapiFAI implements RestoreDomainType {

	private static final Logger logger = LoggerFactory.getLogger(RestoreMapiFAI.class);

	private static final ValueReader<ItemValue<MapiFAI>> faiReader = JsonUtils
			.reader(new TypeReference<ItemValue<MapiFAI>>() {
			});

	private final RestoreLogger log;
	private ItemValue<Domain> domain;
	private IServiceProvider target;

	public RestoreMapiFAI(RestoreLogger log, ItemValue<Domain> domain, IServiceProvider target) {
		this.log = log;
		this.domain = domain;
		this.target = target;
	}

	@Override
	public String type() {
		return "mapi_fai";
	}

	@Override
	public void restore(RecordKey key, String payload) {
		restoreFai(key, payload);
	}

	private void restoreFai(RecordKey key, String payload) {
		String replica = key.uid.replace("mapi_fai_", "");
		IMapiFolderAssociatedInformation fais = target.instance(IMapiFolderAssociatedInformation.class, replica);
		if (Operation.isDelete(key)) {
			fais.deleteByIds(List.of(key.id));
		} else {
			ItemValue<MapiFAI> faiItem = faiReader.read(payload);
			ItemValue<MapiFAI> currentByUid = fais.lookup(faiItem.uid);
			ItemValue<MapiFAI> current = fais.getCompleteById(faiItem.internalId);
			Item kaf = faiItem.item();
			Item lookup = Optional.ofNullable(currentByUid).map(i -> i.item()).orElse(null);
			Item byId = Optional.ofNullable(current).map(i -> i.item()).orElse(null);
			logger.info("FAI kaf: {}, lookup: {}, byItemId: {}", kaf, lookup, byId);
			if (currentByUid != null && currentByUid.internalId != faiItem.internalId) {
				logger.info("FAI del by lookup (id missmatch)");
				fais.deleteByIds(List.of(currentByUid.internalId));
			}

			if (current != null && !current.uid.equals(faiItem.uid)) {
				logger.info("FAI del by id (uid missmatch)");
				fais.deleteByIds(List.of(current.internalId));
				current = null;
			}
			List<ItemValue<MapiFAI>> siblings = fais.getByFolderId(faiItem.value.folderId);
			String kafClass = msgClass(faiItem);
			List<Long> toDel = siblings.stream().filter(iv -> kafClass.equals(msgClass(iv))).map(iv -> iv.internalId)
					.toList();
			logger.info("FAI clean {} siblings with similar message class", toDel.size());
			fais.deleteByIds(toDel);
			if (!toDel.isEmpty()) {
				current = null;
			}

			logger.info("FAI restore isCreate: {}", current == null);
			fais.restore(faiItem, current == null);
		}
	}

	private String msgClass(ItemValue<MapiFAI> iv) {
		JsonObject js = new JsonObject(iv.value.faiJson).getJsonObject("setProperties");
		String klass = js.getString("PidTagMessageClass");
		if (klass == null || klass.isBlank()) {
			klass = js.getString("PidTagSubject");
		}
		return klass;
	}

}
