/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2021
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.backup.continuous.restore;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;

import io.netty.util.concurrent.DefaultThreadFactory;
import io.vertx.core.json.JsonObject;
import net.bluemind.backend.mail.replica.api.IInternelMessageBodyPurgeQueue;
import net.bluemind.core.api.Regex;
import net.bluemind.core.backup.continuous.DataElement;
import net.bluemind.core.backup.continuous.IBackupReader;
import net.bluemind.core.backup.continuous.ILiveBackupStreams;
import net.bluemind.core.backup.continuous.ILiveStream;
import net.bluemind.core.backup.continuous.IRecordStarvationStrategy.ExpectedBehaviour;
import net.bluemind.core.backup.continuous.api.CloneDefaults;
import net.bluemind.core.backup.continuous.dto.CoreTok;
import net.bluemind.core.backup.continuous.index.IIndexBuilder.ReMapped;
import net.bluemind.core.backup.continuous.index.ISerde;
import net.bluemind.core.backup.continuous.index.IStreamLiveIndex;
import net.bluemind.core.backup.continuous.restore.domains.DomainRestorationHandler;
import net.bluemind.core.backup.continuous.restore.domains.RestoreState;
import net.bluemind.core.backup.continuous.restore.index.IndexKeys;
import net.bluemind.core.backup.continuous.restore.index.Indices;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreApiKeys;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreContainerItemIdSeq;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreDomains;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreDpRetention;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreEasSync;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreJobPlans;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreSysconf;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreToken;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreTopology;
import net.bluemind.core.backup.continuous.restore.orphans.RestoreTopology.PromotingServer;
import net.bluemind.core.backup.continuous.restore.queues.RestoreQueue;
import net.bluemind.core.backup.continuous.store.ITopicStore.IResumeToken;
import net.bluemind.core.backup.store.kafka.KafkaToken;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.TaskUtils;
import net.bluemind.directory.api.IDirEntryMaintenance;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.directory.api.RepairConfig;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.system.api.CloneConfiguration;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.sysconf.helper.SysConfHelper;

public class InstallFromBackupTask extends BlockingServerTask implements IServerTask {

	private static final Logger logger = LoggerFactory.getLogger(InstallFromBackupTask.class);

	private final String sourceMcastId;
	private final IBackupReader backupStore;
	private final TopologyMapping topologyMapping;
	private final IServiceProvider target;
	private final Map<String, IResumeToken> processedStreams;

	private final SysconfOverride confOver;

	private final CloneConfiguration cloneConf;

	@VisibleForTesting
	public InstallFromBackupTask(CloneConfiguration conf, IBackupReader store, SysconfOverride over,
			TopologyMapping map, IServiceProvider target) {
		this.sourceMcastId = conf.sourceInstallationId;
		this.cloneConf = conf;
		this.target = target;
		this.processedStreams = new HashMap<>();
		this.topologyMapping = map;
		this.backupStore = store;
		this.confOver = over;
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {

		monitor.begin(100, "Topology, domains then directories...");
		System.setProperty(CloneDefaults.WORKERS_SYSPROP, "" + cloneConf.cloneWorkers);

		Path cloneStatePath = Paths.get(CloneDefaults.CLONE_STATE_PATH);

		ILiveBackupStreams streams = backupStore.forInstallation(sourceMcastId);
		ILiveStream orphansStream = streams.orphans();
		CloneState cloneState = new CloneState(cloneStatePath, orphansStream);

		try {
			ClonedOrphans orphans = cloneOrphans(monitor.subWork(1), orphansStream, cloneState);

			cloneContainerItemIdSeq(monitor, orphansStream, orphans, cloneState);

			logger.info("cloning domains {}", orphans);
			cloneDomains(monitor.subWork(99), streams, cloneState, orphans);
		} catch (Throwable e) { // NOSONAR
			logger.error("cloning error", e);
			preventRestart();

			monitor.end(false, e.getMessage(), "FAILED");
			cloneState.terminate();
			System.exit(1);
		}
	}

	private void preventRestart() {
		// ensure we can't restart as running
		try {
			Files.touch(new File("/etc/bm/bm-core.disabled"));
		} catch (IOException ioe) {
			logger.warn(ioe.getMessage(), ioe);
		}
	}

	public static class ClonedOrphans {

		public final Map<String, PromotingServer> topology;
		public final Map<String, ItemValue<Domain>> domains;
		public final SystemConf sysconf;
		public final CoreTok token;
		public final RestoreContainerItemIdSeq restoreSeq;
		public final RestoreToken restoreTok;

		public ClonedOrphans(Map<String, PromotingServer> topology, Map<String, ItemValue<Domain>> domains,
				SystemConf sysconf, CoreTok coreTok, RestoreContainerItemIdSeq restoreSeq, RestoreToken restoreTok) {
			this.topology = topology;
			this.domains = domains;
			this.sysconf = sysconf;
			this.token = coreTok;
			this.restoreSeq = restoreSeq;
			this.restoreTok = restoreTok;
		}
	}

	private ClonedOrphans cloneOrphans(IServerTaskMonitor monitor, ILiveStream orphansStream, CloneState cloneState) {
		monitor.begin(3, "Cloning orphans (cross-domain data) of installation " + sourceMcastId);
		Map<String, List<DataElement>> orphansByType = new ConcurrentHashMap<>();
		RestoreEasSync restoreEasSync = new RestoreEasSync(monitor);
		Map<String, Consumer<DataElement>> orphanStreamProcessors = Map.of( //
				"easclientid", restoreEasSync::restoreClientId, //
				"easfoldersync", restoreEasSync::restoreFolderSync, //
				"easheatbeat", restoreEasSync::restoreHeatbeat, //
				"easreset", restoreEasSync::restoreReset //
		);

		IResumeToken prevState = cloneState.forTopic(orphansStream);
		monitor.log("IGNORE prevState for " + orphansStream + " -> " + prevState);
		IResumeToken orphansStreamIndex = orphansStream.subscribe(null, de -> {
			Consumer<DataElement> streamProcessor = orphanStreamProcessors.get(de.key.type);
			if (streamProcessor != null) {
				streamProcessor.accept(de);
			} else {
				orphansByType.computeIfAbsent(de.key.type, key -> new ArrayList<>()).add(de);
			}
		});

		RestoreToken restoreTok = new RestoreToken(target);
		CoreTok coreTok = restoreTok.restore(monitor, orphansByType.getOrDefault("installation", new ArrayList<>()));
		Map<String, PromotingServer> topology = new RestoreTopology(target, topologyMapping).restore(monitor,
				orphansByType.getOrDefault("installation", new ArrayList<>()));
		RestoreContainerItemIdSeq restoreSeq = new RestoreContainerItemIdSeq(topology.values());
		restoreSeq.restore(monitor, orphansByType.getOrDefault("container_item_id_seq", new ArrayList<>()));

		Map<String, ItemValue<Domain>> domains = new RestoreDomains(target, topology.values()).restore(monitor,
				orphansByType.getOrDefault("domains", new ArrayList<>()));
		restoreTok.cloneAdmin0Password(coreTok);

		SystemConf sysconf = new RestoreSysconf(target, confOver).restore(monitor,
				orphansByType.getOrDefault("sysconf", new ArrayList<>()));

		waitForSysconfPropagation(monitor, sysconf);

		new RestoreDpRetention(target).restore(monitor, orphansByType.getOrDefault("dp", new ArrayList<>()));
		new RestoreJobPlans(target).restore(monitor, orphansByType.getOrDefault("job_plans", new ArrayList<>()));
		new RestoreApiKeys().restore(monitor, orphansByType.getOrDefault("apikeys", new ArrayList<>()));

		recordProcessed(monitor, cloneState, orphansStream, orphansStreamIndex);
		orphansByType.clear();
		monitor.end(true, "Orphans cloned", null);
		return new ClonedOrphans(topology, domains, sysconf, coreTok, restoreSeq, restoreTok);
	}

	private void waitForSysconfPropagation(IServerTaskMonitor monitor, SystemConf sysconf) {
		Supplier<String> sysconfSig = () -> confHash("in-core", sysconf);
		Supplier<String> sharedSig = () -> confHash("in-rdis", SysConfHelper.fromSharedMap());
		while (!sharedSig.get().equals(sysconfSig.get())) {
			monitor.log("Waiting for sysconf replication to keydb...");
			try {
				Thread.sleep(500);
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				break;
			}
		}
	}

	private String confHash(String kind, SystemConf sysconf) {
		String scKey = sysconf.values.keySet().stream()//
				.filter(s -> !"version".equals(s))//
				.filter(s -> !"external-url".equals(s))//
				.sorted()//
				.collect(Collectors.joining(","));
		logger.warn("SC {}: {}", kind, scKey);
		return Hashing.murmur3_128().hashBytes(scKey.getBytes()).toString();
	}

	private void cloneContainerItemIdSeq(IServerTaskMonitor monitor, ILiveStream orphansStream, ClonedOrphans orphans,
			CloneState cloneState) {
		RestoreContainerItemIdSeq idSeq = orphans.restoreSeq;
		ExecutorService orphanTrackerPool = Executors
				.newSingleThreadExecutor(new DefaultThreadFactory("orphan-tracker"));
		CompletableFuture<IResumeToken> orphanTrack = CompletableFuture.supplyAsync(() -> {
			return orphansStream.subscribe(cloneState.forTopic(orphansStream), de -> {
				if (de.key.valueClass.equals(CoreTok.class.getCanonicalName())) {
					orphans.restoreTok.restore(monitor, Arrays.asList(de));
				} else if (de.key.type.equals("container_item_id_seq")) {
					idSeq.restore(monitor, Arrays.asList(de));
				}
			}, infos -> (cloneState.isTerminated()) ? ExpectedBehaviour.ABORT : ExpectedBehaviour.RETRY);
		}, orphanTrackerPool);
		orphanTrack.whenComplete((v, ex) -> orphanTrackerPool.shutdown());
	}

	private void cloneDomains(IServerTaskMonitor monitor, ILiveBackupStreams streams, CloneState cloneState,
			ClonedOrphans orphans) {
		// exclude, at least, "<mcastid>-crd-dir-entries" created by bm-crp kafka stream
		List<ILiveStream> domainStreams = streams.domains().stream().filter(d -> Regex.DOMAIN.validate(d.domainUid()))
				.toList();
		List<ILiveStream> queueStreams = streams.domains().stream().filter(d -> d.domainUid().equals("sync.q"))
				.toList();

		monitor.log("Filtered domain(s) lists containers {} stream(s) ({}) out of {}", domainStreams.size(),
				domainStreams.stream().map(ILiveStream::domainUid).toList(),
				streams.domains().stream().map(ILiveStream::domainUid).toList());

		monitor.begin(domainStreams.size(), "Cloning domains");
		int goal = domainStreams.size() + queueStreams.size();
		ExecutorService clonePool = Executors.newFixedThreadPool(goal + 1,
				new DefaultThreadFactory("backup-continuous-restore"));

		RecordStarvationHandler starvation = new RecordStarvationHandler(monitor, cloneConf, orphans, target,
				cloneState);

		IInternelMessageBodyPurgeQueue purgeQueueApi = target.instance(IInternelMessageBodyPurgeQueue.class);
		purgeQueueApi.disableReplicationTriggers();

		CompletableFuture<?>[] toWait = new CompletableFuture<?>[goal];
		int slot = 0;

		for (ILiveStream qStream : queueStreams) {
			logger.info("Starting domain stream sync.q");
			/* queue sync (t_message_body_purge_queue) */
			toWait[slot++] = CompletableFuture.supplyAsync(() -> {
				IServerTaskMonitor queueMonitor = monitor.subWork(qStream.domainUid(), 1);
				IResumeToken stateForTopic = cloneState.forTopic(qStream);
				IResumeToken domainStreamIndex = stateForTopic;
				JsonObject prevStateJson;
				if (stateForTopic == null) {
					prevStateJson = new JsonObject()
							.put("group", "clone-syncq-" + UUID.randomUUID().toString().replace("-", ""))
							.put("workers", 4);
				} else {
					prevStateJson = stateForTopic.toJson();
				}
				IResumeToken toUseToken = new KafkaToken(prevStateJson.getString("group"),
						Math.max(prevStateJson.getInteger("workers", 4), 4));
				monitor.log("prevState for " + qStream + " => " + toUseToken);
				try {
					RestoreQueue rq = new RestoreQueue(monitor, target);
					domainStreamIndex = qStream.subscribe(toUseToken, rq::handle, starvation);
				} finally {
					recordProcessed(queueMonitor, cloneState, qStream, domainStreamIndex);
					queueMonitor.end(true, "Queue sync.q fully restored", null);
				}
				return null;
			}, clonePool);
		}

		// Read CRP, get active dirEntries first (ignore dirEntries not in CRP)
		for (ILiveStream domainStream : domainStreams) {
			ItemValue<Domain> domain = orphans.domains.get(domainStream.domainUid());
			if (domain == null) {
				if (logger.isErrorEnabled()) {
					logger.error("domain uid={} not found in orphans", domainStream.domainUid());
				}
				monitor.end(false,
						"Failed to restore " + domainStream.domainUid() + ": " + "domain not found in orphans", null);
				break;
			}
			toWait[slot++] = cloneDomain(monitor, clonePool, domainStream, domain, streams.preSyncForDomain(domain.uid),
					orphans.topology, cloneState, starvation);
		}
		CompletableFuture<Void> globalProm = CompletableFuture.allOf(toWait);
		monitor.log("Waiting for domains cloning global promise {}...", globalProm);
		globalProm.join();
		cloneState.terminate();
		purgeQueueApi.enableReplicationTriggers();
	}

	public CompletableFuture<Void> cloneDomain(IServerTaskMonitor monitor, ExecutorService clonePool,
			ILiveStream domainStream, ItemValue<Domain> domain, Optional<ILiveStream> preSync,
			Map<String, PromotingServer> topology, CloneState cloneState, RecordStarvationHandler starvation) {

		IServerTaskMonitor domainMonitor = monitor.subWork(domain.value.defaultAlias, 1);
		domainMonitor.log("supplyAsync for {} on pool {}", domain.value.defaultAlias, clonePool);
		return CompletableFuture.runAsync(
				() -> syncCloneDomain(domainMonitor, preSync, domainStream, domain, topology, cloneState, starvation),
				clonePool);

	}

	private void syncCloneDomain(IServerTaskMonitor domainMonitor, Optional<ILiveStream> preSync,
			ILiveStream domainStream, ItemValue<Domain> domain, Map<String, PromotingServer> topology,
			CloneState cloneState, RecordStarvationHandler starvation) {
		domainMonitor.begin(1, "Working on domain " + domain.uid);

		ISystemConfiguration confApi = target.instance(ISystemConfiguration.class);
		SystemConf values = confApi.getValues();
		Indices indices = createTopicIndices(domainMonitor, domainStream, domain,
				target.instance(IDirectory.class, domain.uid));

		IResumeToken domainStreamIndex = cloneState.forTopic(domainStream);
		boolean bodiesMigratedInDedicatedTopic = values.booleanValue(SysConfKeys.kafka_bodies_migrated.name(), false);

		try (RestoreState state = new RestoreState(domain.uid, topology)) {
			logger.info("on {} with state {}", domain.uid, state);
			DomainRestorationHandler restoration = new DomainRestorationHandler(domainMonitor,
					cloneConf.skippedContainerTypes, domain, target, starvation, state, indices,
					bodiesMigratedInDedicatedTopic);
			IResumeToken prevState = cloneState.forTopic(domainStream);
			domainMonitor.log("prevState for " + domainStream + " => " + prevState);

			repairMailboxes(domain, state);

			// sync with __presync topic first (repair)
			preSync.ifPresentOrElse(preSyncStream -> {
				domainMonitor.log("presync stream available for {}: launching presync first", domain.uid);
				logger.info("presync stream available for {}: launching presync first", domain.uid);
				if (prevState == null) {
					preSyncStream.subscribe(restoration::handle);
				} else {
					domainMonitor.log("presync available, but previous state is non null: no presync needed");
					logger.info("presync available, but previous state is non null: no presync needed");
				}
			}, () -> logger.info("presync stream *NOT* available for {}", domain.uid));
			logger.info("sub to {}", domainStream);
			domainStreamIndex = domainStream.subscribe(prevState, restoration::handle, starvation);
		} catch (IOException e) {
			logger.error("unexpected error when closing", e);
			domainMonitor.end(false, "Fail to restore " + domain.uid + ": " + e.getMessage(), null);
		} catch (Throwable e) { // NOSONAR catch more ?
			logger.error("unexpected error", e);
			domainMonitor.end(false, "Fail to restore " + domain.uid + ": " + e.getMessage(), null);
		} finally {
			recordProcessed(domainMonitor, cloneState, domainStream, domainStreamIndex);
			domainMonitor.end(true, "Domain " + domain.uid + " fully restored", null);
			indices.close();
		}
	}

	private void repairMailboxes(ItemValue<Domain> domain, RestoreState state) {
		List<ItemValue<Mailbox>> currentMailboxList = target.instance(IMailboxes.class, domain.uid).list();
		currentMailboxList.forEach(ivmbox -> state.storeMailbox(ivmbox.uid, ivmbox));

		long failedRepairs = currentMailboxList.parallelStream().map(mbox -> {
			IDirEntryMaintenance repair = target.instance(IDirEntryMaintenance.class, domain.uid, mbox.uid);
			RepairConfig rc = new RepairConfig();
			rc.dry = false;
			rc.logToCoreLog = false;
			rc.opIdentifiers = Set.of("containers.sharding.location");
			rc.verbose = false;
			return TaskUtils.wait(target, repair.repair(rc)).state;
		}).filter(status -> !status.succeed).count();
		logger.info("Repaired sharding of {} mailboxe(s) -> {} failure(s)", currentMailboxList.size(), failedRepairs);
	}

	private Indices createTopicIndices(IServerTaskMonitor monitor, ILiveStream domainStream, ItemValue<Domain> domain,
			IDirectory dirApi) {
		Stopwatch chrono = Stopwatch.createStarted();
		byte[] empty = new byte[0];
		CompletableFuture<IStreamLiveIndex<String, byte[]>> deletionsIndexBuild = domainStream
				.index("deletions-of-" + domain.uid.replace(".internal", ""), idxBuilder -> idxBuilder//
						.flatMap((k, v) -> {
							if (v == null) {
								return List.of(new ReMapped<>(IndexKeys.deletion(k), empty));
							} else if ("delete".equalsIgnoreCase(k.operation)) {
								return List.of(new ReMapped<>(IndexKeys.deletion(k), empty),
										new ReMapped<>(IndexKeys.deletion(k.uid, v), empty));
							} else if (k.valueClass != null && k.valueClass.contains("MailboxRecord")) {
								// create or update, generate a null deletion key to deal with resurrected
								// itemIds...
								return List.of(new ReMapped<>(IndexKeys.deletion(k), null),
										new ReMapped<>(IndexKeys.deletion(k.uid, v), null));
							} else {
								return Collections.emptyList();
							}
						}), //
						ISerde.RECORD_KEY, ISerde.jsonStringField("uid"), ISerde.UTF8_STRING, ISerde.PRISTINE);

		IStreamLiveIndex<String, byte[]> delIndex = deletionsIndexBuild.orTimeout(4, TimeUnit.HOURS).join();

		CompletableFuture<IStreamLiveIndex<String, JsonObject>> dirIndexBuild = domainStream
				.index("directory-by-uid-of-" + domain.uid.replace(".internal", ""), idxBuilder -> idxBuilder//
						.filter((k, v) -> v != null && "dir".equals(k.type)
								&& "net.bluemind.directory.service.DirEntryAndValue".equals(k.valueClass)
								&& !"delete".equalsIgnoreCase(k.operation))
						.filter((k, v) -> {
							// use the deletion index to skip removed dir entries
							String delKey = IndexKeys.deletion(k.uid, v.getString("uid"));
							return delIndex.get(delKey) == null;
						}).map((k, v) -> new ReMapped<>(v.getString("uid"), v)), //
						ISerde.RECORD_KEY, ISerde.JSON, ISerde.UTF8_STRING, ISerde.JSON);

		IStreamLiveIndex<String, JsonObject> dirIndex = dirIndexBuild.orTimeout(4, TimeUnit.HOURS).join();
		monitor.log("Indexing phase completed for {} in {}sec", domain.uid, chrono.elapsed(TimeUnit.SECONDS));
		return new Indices(domain, dirApi, Set.of("system", "addressbook_" + domain.uid), delIndex, dirIndex);
	}

	private void recordProcessed(IServerTaskMonitor monitor, CloneState cloneState, ILiveStream stream,
			IResumeToken index) {
		monitor.log("Processed " + stream + " up to " + index);
		processedStreams.put(stream.domainUid(), index);
		cloneState.track(stream.fullName(), index).save();
	}
}
