package net.bluemind.delivery.rules;

import static java.util.Arrays.asList;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleCondition.contains;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleCondition.equal;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleCondition.exists;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleCondition.not;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterContains.Comparator.FULLSTRING;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterContains.Comparator.PREFIX;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterContains.Comparator.SUBSTRING;
import static net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterContains.Modifier.CASE_INSENSITIVE;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.apache.james.mime4j.codec.DecodeMonitor;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.address.AddressList;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.dom.address.MailboxList;
import org.apache.james.mime4j.dom.field.UnstructuredField;
import org.apache.james.mime4j.field.UnstructuredFieldImpl;
import org.apache.james.mime4j.stream.RawField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.buffer.Buffer;
import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.flags.WellKnownFlags;
import net.bluemind.backend.mail.replica.api.AppendTx;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.core.api.Stream;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.core.sendmail.ISendmail;
import net.bluemind.delivery.lmtp.common.DeliveryContent;
import net.bluemind.delivery.lmtp.common.DeliveryContent.DeferredActionMessage;
import net.bluemind.delivery.lmtp.common.FreezableDeliveryContent;
import net.bluemind.delivery.lmtp.common.FreezableDeliveryContent.SerializedMessage;
import net.bluemind.delivery.lmtp.common.IDeliveryContext;
import net.bluemind.delivery.lmtp.common.ResolvedBox;
import net.bluemind.delivery.rules.auditlog.RuleAuditLogMapper;
import net.bluemind.delivery.rules.auditlog.RuleEngineAuditLogService;
import net.bluemind.delivery.smtp.ndr.SendmailCredentials;
import net.bluemind.delivery.smtp.ndr.SendmailHelper;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirEntryPath;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.eclipse.common.RunnableExtensionLoader;
import net.bluemind.mailbox.api.rules.FieldValueProvider;
import net.bluemind.mailbox.api.rules.MailFilterRule;
import net.bluemind.mailbox.api.rules.MailFilterRule.Trigger;
import net.bluemind.mailbox.api.rules.ParameterValueProvider;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleAction;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionAddHeaders;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionCategorize;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionCopy;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionCustom;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionDeferredAction;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionMarkAsDeleted;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionMarkAsImportant;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionMarkAsRead;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionMove;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionPrioritize;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionRedirect;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionRemoveHeaders;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionReply;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionSetFlags;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionTransfer;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionUncategorize;
import net.bluemind.mailbox.api.rules.actions.MailFilterRuleActionUnfollow;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleCondition;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterContains.Modifier;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;

public class RuleEngine {
	private static final Logger logger = LoggerFactory.getLogger(RuleEngine.class);

	private final IDeliveryContext ctx;
	private final ISendmail mailer;
	private final DeliveryContent content;
	private final FieldValueProvider fieldValueProvider;
	private final ParameterValueProvider parameterValueProvider;
	private final MailboxVacationSendersCache.Factory vacationCacheFactory;

	private Supplier<Optional<RuleEngineAuditLogService>> auditLogServiceSupplier;

	public static final List<String> mailingListHeader = asList("list-id", "list-help", "list-subscribe",
			"list-unsubscribe", "list-post", "list-owner", "list-archive").stream().map(h -> "headers." + h).toList();

	private static final List<MailFilterRuleCondition> vacationExtraConditions = asList(
			not(contains("from.email", asList("MAILER-DAEMON", "LISTSERV", "majordomo", "noreply", "no-reply"), PREFIX,
					CASE_INSENSITIVE) //
					.or(contains("from.email", "-request@", SUBSTRING, Modifier.NONE))
					.or(contains("from.email", "owner-", PREFIX, Modifier.NONE))), //
			not(contains("from.email", "BM_DYNAMIC_ADDRESSES_ME")),
			not(contains("headers.precedence", asList("bulk", "list", "junk"), FULLSTRING, CASE_INSENSITIVE)), //
			not(exists("headers.auto-submitted"))
					.or(contains("headers.auto-submitted", "no", FULLSTRING, CASE_INSENSITIVE)), //
			not(exists("headers.x-ignorevacation"))
					.or(contains("headers.x-ignorevacation", "no", FULLSTRING, CASE_INSENSITIVE)), //
			not(exists(mailingListHeader)), //
			not(contains("headers.x-dspam-result", "spam", FULLSTRING, CASE_INSENSITIVE)),
			not(contains("headers.x-spam-flag", "yes", FULLSTRING, CASE_INSENSITIVE)),
			equal(asList("to.email", "cc.email", "bcc.email"), asList("BM_DYNAMIC_ADDRESSES_ME")) //
					.or(contains(asList("headers.resent-to", "headers.resent-cc", "headers.resent-bcc"),
							"BM_DYNAMIC_ADDRESSES_ME")));

	private static final List<IMailFilterRuleCustomAction> CUSTOM_ACTIONS = load();

	private static List<IMailFilterRuleCustomAction> load() {
		RunnableExtensionLoader<IMailFilterRuleCustomAction> rel = new RunnableExtensionLoader<>();
		return rel.loadExtensionsWithPriority("net.bluemind.delivery.rules", "action", "action", "impl");
	}

	public RuleEngine(IDeliveryContext ctx, ISendmail mailer, DeliveryContent content,
			MailboxVacationSendersCache.Factory vacationCacheFactory) {
		this.ctx = ctx;
		this.mailer = mailer;
		this.content = content;
		this.fieldValueProvider = new FieldValueMessageProvider(content.message(), content.size(),
				content.mailboxRecord());
		this.parameterValueProvider = new ParameterValueCoreProvider(content.box(), ctx.provider());
		this.vacationCacheFactory = vacationCacheFactory;
		RuleAuditLogMapper mapper = new RuleAuditLogMapper();
		RuleEngineAuditLogService auditLogService = new RuleEngineAuditLogService(ctx.provider(), mapper);
		this.auditLogServiceSupplier = () -> {
			if (StateContext.getState().equals(SystemState.CORE_STATE_RUNNING)) {
				return Optional.of(auditLogService);
			}
			return Optional.empty();
		};
	}

	public DeliveryContent content() {
		return content;
	}

	private ResolvedBox originalBox() {
		return content.box();
	}

	private MailboxRecord originalRecord() {
		return content.mailboxRecord();
	}

	private Message originalMessage() {
		return content.message();
	}

	private String originalSubtree() {
		return IMailReplicaUids.subtreeUid(originalBox().dom.uid, originalBox().mbox);
	}

	private String partition(ResolvedBox box) {
		return CyrusPartition.forServerAndDomain(box.entry.dataLocation, box.dom.uid).name;
	}

	public RuleEngine apply(List<MailFilterRule> rules) {
		addVacationSpecificConditions(rules);
		DeliveryContent newContent = applyRulesActions(matchingRules(rules));
		return new RuleEngine(ctx, mailer, newContent, vacationCacheFactory);
	}

	private void addVacationSpecificConditions(List<MailFilterRule> rules) {
		rules.stream() //
				.filter(rule -> MailFilterRule.Type.VACATION.equals(rule.type)) //
				.findFirst() //
				.ifPresent(vacation -> {
					// Vacation is not active or dates doesn't match: we can clear this box cache.
					if (!vacation.active || !vacation.match(fieldValueProvider, parameterValueProvider)) {
						vacationCacheFactory.clear(originalBox().mbox.uid);
					}
					vacation.conditions.addAll(vacationExtraConditions);
				});
	}

	private List<MailFilterRule> matchingRules(List<MailFilterRule> rules) {
		List<MailFilterRule> filtered = rules.stream() //
				.filter(rule -> rule.active && rule.trigger == Trigger.IN
						&& rule.match(fieldValueProvider, parameterValueProvider))
				.toList();
		if (!filtered.isEmpty()) {
			logger.info("[rules] {} out of {} rule(s) are matching", filtered.size(), rules.size());
		}
		return filtered;
	}

	private DeliveryContent applyRulesActions(List<MailFilterRule> rules) {
		if (!rules.isEmpty()) {
			logger.info("[rules] Applying {} rules on {}", rules.size(), content);
		}
		return rules.stream() //
				.sequential() //
				.reduce(content, this::applyRuleActions, (result1, result2) -> result2);
	}

	private DeliveryContent applyRuleActions(DeliveryContent previousContent, MailFilterRule rule) {
		if (previousContent.isEmpty() || previousContent.stop()) {
			String cause = previousContent.isEmpty() ? "message discarded" : "previous action ask to stop";
			logger.info("[rules] stop applying rule on {}: {}", content, cause);
			return previousContent;
		}
		logger.info("[rules] applying rule ({} actions) on {}", rule.actions.size(), content);
		DeliveryContent nextContent = previousContent.withStop(rule.stop);
		return rule.actions.stream().sequential() //
				.reduce(nextContent, //
						(deliveryContent, action) -> applyAction(deliveryContent, rule, action), //
						(result1, result2) -> result2);
	}

	public record RuleActionAndContent(DeliveryContent nextContent, MailFilterRuleAction action) {

	}

	private DeliveryContent applyAction(DeliveryContent nextContent, MailFilterRule rule, MailFilterRuleAction action) {

		logger.info("[rules] applying rule action {} on {}", action.name, content);
		auditLogServiceSupplier.get().ifPresent(auditlog -> auditlog
				.logCreate(new RuleActionAndContent(nextContent, action), nextContent.box().dom.uid));

		return switch (action.name) {
		case ADD_HEADER -> addHeaders(nextContent, (MailFilterRuleActionAddHeaders) action);
		case CATEGORIZE -> addHeaders(nextContent, (MailFilterRuleActionCategorize) action);
		case COPY -> copy(nextContent, (MailFilterRuleActionCopy) action);
		case DEFERRED_ACTION -> deferredAction(nextContent, rule, (MailFilterRuleActionDeferredAction) action);
		case DISCARD -> nextContent.withoutMessage();
		case MARK_AS_DELETED -> setFlags(nextContent, (MailFilterRuleActionMarkAsDeleted) action);
		case MARK_AS_IMPORTANT -> setFlags(nextContent, (MailFilterRuleActionMarkAsImportant) action);
		case MARK_AS_READ -> setFlags(nextContent, (MailFilterRuleActionMarkAsRead) action);
		case MOVE -> move(nextContent, (MailFilterRuleActionMove) action);
		case PRIORITIZE -> addHeaders(nextContent, (MailFilterRuleActionPrioritize) action);
		case REDIRECT -> redirect(nextContent, (MailFilterRuleActionRedirect) action);
		case REMOVE_HEADERS -> removeHeaders(nextContent, (MailFilterRuleActionRemoveHeaders) action);
		case REPLY -> reply(nextContent, (MailFilterRuleActionReply) action, rule);
		case SET_FLAGS -> setFlags(nextContent, (MailFilterRuleActionSetFlags) action);
		case TRANSFER -> transfer(nextContent, (MailFilterRuleActionTransfer) action);
		case UNCATEGORIZE -> removeHeaders(nextContent, (MailFilterRuleActionUncategorize) action);
		case UNFOLLOW -> removeHeaders(nextContent, (MailFilterRuleActionUnfollow) action);
		case CUSTOM -> customAction(nextContent, (MailFilterRuleActionCustom) action);
		};
	}

	private DeliveryContent addHeaders(DeliveryContent nextContent, MailFilterRuleActionAddHeaders addHeaders) {
		addHeaders.headers.forEach((name, value) -> {
			RawField raw = new RawField(name, value);
			UnstructuredField parsed = UnstructuredFieldImpl.PARSER.parse(raw, DecodeMonitor.SILENT);
			logger.info("[rules] adding header {}:{} [{}]", name, value, nextContent);
			originalMessage().getHeader().addField(parsed);
		});
		return nextContent;
	}

	private DeliveryContent copy(DeliveryContent nextContent, MailFilterRuleActionCopy copy) {
		String subtree;
		ResolvedBox box;
		if (copy.subtree().equals("user")) {
			subtree = originalSubtree();
			box = originalBox();
		} else {
			subtree = copy.subtree();
			box = subtreeToBox(subtree);
		}

		if (box == null) {
			return nextContent;
		}

		String partition = partition(box);
		IDbReplicatedMailboxes treeApi = ctx.provider().instance(IDbByContainerReplicatedMailboxes.class, subtree);
		ItemValue<MailboxReplica> copyFolder = treeApi.byReplicaName(copy.folder());
		try {
			FreezableDeliveryContent copiedContent = FreezableDeliveryContent.copy(nextContent);
			SerializedMessage serializedMessage = copiedContent.serializedMessage();

			logger.info("[rules] copying into {} [{}]", copyFolder.value, copiedContent.content());

			IDbMessageBodies bodiesUpload = ctx.provider().instance(IDbMessageBodies.class, partition);
			Stream stream = VertxStream.stream(Buffer.buffer(serializedMessage.buffer()));
			bodiesUpload.create(serializedMessage.guid(), stream);

			AppendTx appendTx = treeApi.prepareAppend(copyFolder.internalId, 1);
			MailboxRecord rec = new MailboxRecord();
			rec.conversationId = originalRecord().conversationId;
			rec.messageBody = originalRecord().messageBody;
			rec.imapUid = appendTx.imapUid;
			rec.flags = new ArrayList<>();
			rec.internalDate = new Date();
			rec.lastUpdated = rec.internalDate;
			rec.conversationId = rec.internalDate.getTime();
			IDbMailboxRecords recs = ctx.provider().instance(IDbMailboxRecords.class, copyFolder.uid);
			recs.create(rec.imapUid + ".", rec);
		} catch (IOException e) {
			logger.error("[rule] failed to serialize message for {}, skipping copy into {}", //
					nextContent, copy.folder());
		}
		return nextContent;
	}

	private DeliveryContent deferredAction(DeliveryContent nextContent, MailFilterRule rule,
			MailFilterRuleActionDeferredAction action) {
		long ruleId = Long.parseLong(rule.clientProperties.get("PidTagRuleId"));
		String ruleProvider = rule.clientProperties.get("PidTagRuleProvider");
		String deferredAction = action.clientProperties.get("remainder");
		logger.info("[rules] DeferredAction owner:{} ruleId:{} provider:{}", nextContent.box().entry.entryUid, ruleId,
				ruleProvider);
		return nextContent.withDeferredAction(new DeferredActionMessage(ruleId, ruleProvider, deferredAction));
	}

	private DeliveryContent setFlags(DeliveryContent nextContent, MailFilterRuleActionSetFlags setFlags) {
		originalRecord().flags = new ArrayList<>(originalRecord().flags);
		setFlags.flags.forEach(flag -> {
			logger.info("[rules] flagging with {} [{}]", flag, nextContent);
			originalRecord().flags.add(WellKnownFlags.resolve(flag));
		});
		setFlags.internalFlags.stream().filter(flag -> flag.equals("\\Expunged")).findFirst().ifPresent(flag -> {
			logger.info("[rules] flagging (internal) with {} [{}]", flag, nextContent);
			originalRecord().internalFlags.add(MailboxRecord.InternalFlag.expunged);
		});
		return nextContent;
	}

	private DeliveryContent move(DeliveryContent nextContent, MailFilterRuleActionMove move) {
		String subtree;
		ResolvedBox box;
		if (move.subtree().equals("user")) {
			subtree = originalSubtree();
			box = originalBox();
		} else {
			subtree = move.subtree();
			box = subtreeToBox(subtree);
		}
		if (box == null) {
			logger.info("[rules] looking for subtree:{} box:null", subtree);
			return nextContent;
		}
		IDbReplicatedMailboxes treeApi = ctx.provider().instance(IDbByContainerReplicatedMailboxes.class, subtree);
		ItemValue<MailboxReplica> newFolder = (move.id() == null) //
				? treeApi.byReplicaName(move.folder())
				: treeApi.getCompleteById(move.id());
		String newFolderName = (newFolder != null) ? newFolder.value.fullName : null;
		logger.info("[rules] moving to {} (id={}, name={}) [{}]", move.folder(), move.id(), newFolderName, nextContent);
		return (newFolder != null) ? nextContent.withFolder(newFolder).withBox(box) : nextContent;
	}

	private record DirEntryPath(String domainUid, String path) {

	}

	private ResolvedBox subtreeToBox(String subtree) {
		DirEntryPath dirEntryPath = subtreeToDirEntryPath(subtree);
		IDirectory dirApi = ctx.provider().instance(IDirectory.class, dirEntryPath.domainUid);
		DirEntry entry = dirApi.getEntry(dirEntryPath.path);
		return ctx.mailboxLookup().lookupEmail(entry.email);
	}

	private DirEntryPath subtreeToDirEntryPath(String subtree) {
		String[] tokens = subtree.split("!");
		String domainUid = tokens[0].replace("subtree_", "").replace("_", ".");
		if (tokens[1].contains("user.")) {
			String path = IDirEntryPath.path(domainUid, tokens[1].replace("user.", ""), Kind.USER);
			return new DirEntryPath(domainUid, path);
		} else {
			String path = IDirEntryPath.path(domainUid, tokens[1], Kind.MAILSHARE);
			return new DirEntryPath(domainUid, path);
		}
	}

	private DeliveryContent redirect(DeliveryContent nextContent, MailFilterRuleActionRedirect redirect) {
		List<Mailbox> mailboxes = redirect.emails().stream().map(email -> SendmailHelper.formatAddress(null, email))
				.toList();
		MailboxList to = new MailboxList(mailboxes, true);
		String from = content.from();
		logger.info("[rules] redirecting to {} (keep copy:{}) [{}]", stringify(mailboxes), redirect.keepCopy,
				nextContent);

		String redirectFor = originalBox().entry.entryUid + "@" + originalBox().dom.uid;
		RawField raw = new RawField("X-BM-redirect-for", redirectFor);
		UnstructuredField parsed = UnstructuredFieldImpl.PARSER.parse(raw, DecodeMonitor.SILENT);
		originalMessage().getHeader().addField(parsed);

		mailer.send(SendmailCredentials.asAdmin0(), from, originalBox().dom.uid, to, originalMessage());
		return (redirect.keepCopy) ? nextContent : nextContent.withoutMessage();
	}

	private DeliveryContent removeHeaders(DeliveryContent nextContent,
			MailFilterRuleActionRemoveHeaders removeHeaders) {
		removeHeaders.headerNames.forEach(name -> {
			logger.info("[rules] removing header {} [{}]", name, nextContent);
			originalMessage().getHeader().removeFields(name);
		});
		return nextContent;
	}

	private DeliveryContent reply(DeliveryContent nextContent, MailFilterRuleActionReply reply, MailFilterRule rule) {
		boolean isVacation = rule.type == MailFilterRule.Type.VACATION;
		if (!isVacation) {
			return doReply(nextContent, reply, isVacation);
		}
		MailboxVacationSendersCache recipients = vacationCacheFactory.get(originalBox().mbox.uid);
		AddressList addressList = originalMessage().getReplyTo();
		String sender = (addressList == null || addressList.isEmpty())
				? originalMessage().getFrom().stream().findFirst().map(Mailbox::getAddress).orElse(null)
				: addressList.flatten().stream().findFirst().map(Mailbox::getAddress).orElse(null);
		return recipients.ifMissingDoGetOrElseGet(sender, () -> {
			logger.info("[rules][vacation] must reply to {} [{}]", sender, nextContent);
			return doReply(nextContent, reply, isVacation);
		}, () -> {
			logger.info("[rules][vacation] skip reply to {} [{}]", sender, nextContent);
			return nextContent;
		});
	}

	private DeliveryContent doReply(DeliveryContent nextContent, MailFilterRuleActionReply reply, boolean isVacation) {
		Message originalMessage = originalMessage();
		MessageCreator creator = new MessageCreator(originalBox(), originalMessage);
		Message replyMessage = creator.newMessageWithOriginalCited(originalMessage().getFrom(), "Re", reply.subject,
				reply.plainBody, reply.htmlBody, true, isVacation);
		if (isVacation) {
			RawField raw = new RawField("X-Autoreply", "yes"); // "Auto-Submitted", "auto-replied"
			replyMessage.getHeader().addField(UnstructuredFieldImpl.PARSER.parse(raw, DecodeMonitor.SILENT));
			// Allow the vacation message to be tracked in mail thread (conversation)
			if (originalMessage.getMessageId() != null && !originalMessage.getMessageId().isBlank()) {
				RawField inReplyTo = new RawField("In-Reply-To", originalMessage.getMessageId());
				replyMessage.getHeader().setField(UnstructuredFieldImpl.PARSER.parse(inReplyTo, DecodeMonitor.SILENT));
			}
		}
		String sender = originalMessage.getFrom().stream().findFirst().map(Mailbox::getAddress).orElse(null);
		logger.info("[rules] replying to {} [{}]", sender, nextContent);
		mailer.send(SendmailCredentials.asAdmin0(), originalBox().dom.uid, replyMessage);

		return nextContent;
	}

	private DeliveryContent transfer(DeliveryContent nextContent, MailFilterRuleActionTransfer transfer) {
		List<Mailbox> mailboxes = transfer.emails.stream().map(email -> SendmailHelper.formatAddress(null, email))
				.toList();
		MailboxList to = new MailboxList(mailboxes, true);
		MessageCreator creator = new MessageCreator(originalBox(), originalMessage());
		Message transferMessage = (!transfer.asAttachment) //
				? creator.newMessageWithOriginalCited(to, "Fwd", null, "", "", false, false) //
				: creator.newMessageWithOriginalAttached(to);
		logger.info("[rules] transferring to {} (keep copy:{}, as attachment:{}) [{}]", stringify(mailboxes),
				transfer.keepCopy, transfer.asAttachment, nextContent);
		mailer.send(SendmailCredentials.asAdmin0(), originalBox().dom.uid, transferMessage);
		return (transfer.keepCopy) ? nextContent : nextContent.withoutMessage();
	}

	private DeliveryContent customAction(DeliveryContent nextContent, MailFilterRuleActionCustom custom) {
		return CUSTOM_ACTIONS.stream() //
				.filter(customAction -> custom.kind != null && custom.kind.equals(customAction.kind())) //
				.findFirst() //
				.map(customAction -> {
					logger.info("[rules] applying a custom action: kind:{}, parameters:{} [{}]", custom.kind,
							custom.parameters, nextContent);
					return customAction.applyTo(nextContent, custom);
				}) //
				.orElse(nextContent);
	}

	private String stringify(List<Mailbox> mailboxes) {
		return mailboxes.stream().map(Mailbox::getAddress).collect(Collectors.joining(","));
	}

}
