package net.bluemind.lib.elasticsearch.allocations.rebalance;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.lib.elasticsearch.IndexAliasMapping;
import net.bluemind.lib.elasticsearch.allocations.AllocationShardStats;
import net.bluemind.lib.elasticsearch.allocations.RingIndexPositionAllocator;

public class RebalanceAliasAllocator implements RingIndexPositionAllocator<Rebalance> {
	private static final Logger logger = LoggerFactory.getLogger(RebalanceAliasAllocator.class);

	private record AliasStat(String name, long docount) {

	}

	public IndexPositionAllocation apply(Rebalance rebalance) {

		rebalance.sources.sort(Comparator.comparingLong(c -> c.docCount));
		AllocationShardStats source = rebalance.sources.getLast();
		logger.info("index {} with docs {} will be splitted", source.indexName, source.docCount);
		Map<String, Long> map = source.mailboxesCount.stream()
				.collect(Collectors.toMap(m -> m.aliasName, m -> m.docCount, Long::sum));
		long meanDocs = rebalance.averageDocCount;
		logger.info("ring contains an average of {} documents", meanDocs);
		List<AliasStat> aliasStats = map.entrySet().stream().map(e -> new AliasStat(e.getKey(), e.getValue()))
				.collect(Collectors.toList());
		Collections.sort(aliasStats, new Comparator<AliasStat>() {
			@Override
			public int compare(AliasStat alias1, AliasStat alias2) {
				String readAliasString = "_ring_alias_read";
				Integer position1 = Integer.parseInt(
						alias1.name.substring(alias1.name.indexOf(readAliasString) + readAliasString.length()));
				Integer position2 = Integer.parseInt(
						alias2.name.substring(alias2.name.indexOf(readAliasString) + readAliasString.length()));
				return position1.compareTo(position2);
			}

		});

		long availableDocs = meanDocs;
		boolean first = true;
		ArrayList<String> aliasesToMove = new ArrayList<String>();
		Iterator<AliasStat> it = aliasStats.iterator();
		while (it.hasNext()) {
			AliasStat stat = it.next();
			logger.info("index '{}': alias '{}' contains {} documents", source.indexName, stat.name, stat.docount);
			if (stat.docount < availableDocs) {
				first = false;
				availableDocs -= stat.docount();
				aliasesToMove.add(stat.name);
			} else if (first) {
				aliasesToMove.add(stat.name);
				first = false;
				break;
			} else {
				break;
			}
		}
		return new IndexPositionAllocation(source.indexName, aliasesToMove.stream()
				.mapToInt(n -> Integer.parseInt(n.replace(IndexAliasMapping.MAILSPOOL_RING_ALIAS_READ_NAME, ""))).max()
				.getAsInt());
	}
}
