/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lib.elasticsearch;

import static java.util.stream.Collectors.joining;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;

import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthSchemeProvider;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.message.BasicHeader;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Platform;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.module.afterburner.AfterburnerModule;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.ByteStreams;
import com.typesafe.config.ConfigException;

import co.elastic.clients.ApiClient;
import co.elastic.clients.elasticsearch.ElasticsearchAsyncClient;
import co.elastic.clients.elasticsearch.ElasticsearchClient;
import co.elastic.clients.elasticsearch._types.ElasticsearchException;
import co.elastic.clients.elasticsearch._types.HealthStatus;
import co.elastic.clients.elasticsearch._types.analysis.Analyzer;
import co.elastic.clients.elasticsearch.indices.PutMappingResponse;
import co.elastic.clients.elasticsearch.indices.resolve_index.ResolveIndexItem;
import co.elastic.clients.json.DelegatingDeserializer;
import co.elastic.clients.json.JsonData;
import co.elastic.clients.json.ObjectDeserializer;
import co.elastic.clients.json.jackson.JacksonJsonpMapper;
import co.elastic.clients.transport.ElasticsearchTransport;
import net.bluemind.configfile.elastic.ElasticsearchConfig;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.lib.elasticsearch.config.ElasticsearchClientConfig;
import net.bluemind.lib.elasticsearch.config.IndexAliasMode;
import net.bluemind.lib.elasticsearch.config.Mode;
import net.bluemind.lib.elasticsearch.exception.ElasticIndexException;
import net.bluemind.network.topology.IServiceTopology;
import net.bluemind.network.topology.Topology;
import net.bluemind.network.utils.NetworkHelper;
import net.bluemind.server.api.TagDescriptor;

public final class ESearchActivator implements BundleActivator {
	private static Logger logger = LoggerFactory.getLogger(ESearchActivator.class);

	private static final String ES_TAG = TagDescriptor.bm_es.getTag();
	private static final Map<String, ElasticsearchTransport> transports = new ConcurrentHashMap<>();
	private static final Map<String, Lock> refreshLocks = new ConcurrentHashMap<>();
	private static final Map<String, IndexDefinition> indexes = new HashMap<>();

	/**
	 * key for {@link #putMeta(String, String, String)}. Indices with this prop will
	 * be ignored when allocating new aliases
	 */
	public static final String BM_MAINTENANCE_STATE_META_KEY = "bmMaintenanceState";

	static {
		System.setProperty("es.set.netty.runtime.available.processors", "false");
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext )
	 */
	@Override
	public void start(BundleContext bundleContext) throws Exception {
		fixupElasticsearchClientSerde();
		loadIndexSchema();
		setupConfigRefreshListener();
		logger.info("ES activator started , schemas : {}", indexes.keySet());
	}

	private static synchronized void setupConfigRefreshListener() {
		ElasticsearchClientConfig.addListener(newConfig -> {
			transports.clear();
		});
	}

	private static void fixupElasticsearchClientSerde() {
		@SuppressWarnings("unchecked")
		ObjectDeserializer<Analyzer> unwrapped = (ObjectDeserializer<Analyzer>) DelegatingDeserializer
				.unwrap(Analyzer._DESERIALIZER);
		unwrapped.setTypeProperty("type", "custom");
	}

	private static void loadIndexSchema() throws IOException, InvalidRegistryObjectException, CoreException {
		IExtensionPoint ep = Platform.getExtensionRegistry().getExtensionPoint("net.bluemind.elasticsearch.schema");
		for (IExtension ext : ep.getExtensions()) {
			for (IConfigurationElement ce : ext.getConfigurationElements()) {
				String index = ce.getAttribute("index");
				String schema = ce.getAttribute("schema");
				// to override the count for faster testing
				int count = Integer.parseInt(System.getProperty("es." + index + ".count", ce.getAttribute("count")));
				ISchemaMatcher matcher = (ce.getAttribute("schemamatcher") != null)
						? (ISchemaMatcher) ce.createExecutableExtension("schemamatcher")
						: null;
				boolean rewritable = Boolean.parseBoolean(ce.getAttribute("rewritable"));
				boolean supportsAliasRing = Boolean.parseBoolean(ce.getAttribute("supportsAliasRing"));
				Bundle bundle = Platform.getBundle(ext.getContributor().getName());
				URL url = bundle.getResource(schema);
				try (InputStream in = url.openStream()) {
					indexes.put(index, new IndexDefinition(index, ByteStreams.toByteArray(in), matcher, count,
							rewritable, supportsAliasRing));
					refreshLocks.put(index, new ReentrantLock());
				}

				if (logger.isDebugEnabled()) {
					logger.debug("schema for index {}: \n {} ", index, new String(indexes.get(index).schema));
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	@Override
	public void stop(BundleContext bundleContext) throws Exception {
		// Nothing to do
	}

	@VisibleForTesting
	public static final void initClient(ElasticsearchTransport transport) {
		transports.put(ES_TAG, transport);
	}

	public static final void initClasspath() {
		ElasticsearchTransport client = initTransport(ES_TAG);
		if (client != null) {
			transports.put(ES_TAG, client);
		} else {
			logger.warn("elasticsearch node not found");
		}
	}

	public static ElasticsearchClient getClient(List<String> hosts, AuthenticationCredential authCred) {
		String hshtxt = hash(hosts, authCred);
		if (hshtxt == null) {
			return null;
		}
		ElasticsearchTransport transport = transports.computeIfAbsent(hshtxt, k -> createTansport(hosts, authCred));
		return buildClient(hshtxt, transport, ElasticsearchClient::new);
	}

	public static ElasticsearchClient getClient() {
		ElasticsearchTransport transport = transports.computeIfAbsent(ES_TAG, ESearchActivator::initTransport);
		return buildClient(ES_TAG, transport, ElasticsearchClient::new);
	}

	public static ElasticsearchAsyncClient getAsyncClient() {
		ElasticsearchTransport transport = transports.computeIfAbsent(ES_TAG, ESearchActivator::initTransport);
		return buildClient(ES_TAG, transport, ElasticsearchAsyncClient::new);
	}

	public static <T extends ApiClient<?, ?>> T buildClient(String tag, ElasticsearchTransport transport,
			Function<ElasticsearchTransport, T> builder) {
		if (transport == null) {
			logger.error("no elasticsearch instance found for tag {}", tag);
			return null;
		} else {
			return builder.apply(transport);
		}
	}

	private static ElasticsearchTransport initTransport(String tag) {
		List<String> hosts = hosts(tag);
		if (hosts == null || hosts.isEmpty()) {
			logger.warn("Es host missing for tag {}", tag);
			return null;
		}
		return createTansport(hosts, new AuthenticationCredential(Authentication.NONE, null, null));
	}

	private static List<String> hosts(String tag) {
		return Topology.getIfAvailable().map(t -> topoHots(t, tag)).orElse(Collections.emptyList());
	}

	private static List<String> topoHots(IServiceTopology topo, String tag) {
		return topo.nodes().stream().filter(iv -> iv.value.tags.contains(tag)).map(iv -> iv.value.address()).toList();
	}

	public static ElasticsearchTransport createTansport(List<String> hosts, AuthenticationCredential authCred) {
		HttpHost[] httpHosts = hosts.stream().map(host -> new HttpHost(host, 9200)).toArray(l -> new HttpHost[l]);
		RestClient restClient;
		try {
			ElasticsearchConfig.Client clientConfig = ElasticsearchConfig.Client.of(ElasticsearchClientConfig.get());

			HttpAsyncClientBuilder httpAsyncClientBuilder = HttpAsyncClientBuilder.create()
					.setDefaultAuthSchemeRegistry(RegistryBuilder.<AuthSchemeProvider>create().build())//
					.setDefaultCredentialsProvider(new BasicCredentialsProvider())//
					.disableAuthCaching()//
					.setMaxConnTotal(clientConfig.pool().maxConnTotal()) //
					.setMaxConnPerRoute(clientConfig.pool().maxConnPerRoute());

			RestClientBuilder restClientBuilder = RestClient.builder(httpHosts) //
					.setRequestConfigCallback(builder -> builder //
							.setConnectTimeout((int) clientConfig.timeout().connect().toMillis()) //
							.setSocketTimeout((int) clientConfig.timeout().socket().toMillis()) //
							.setConnectionRequestTimeout((int) clientConfig.timeout().request().toMillis()));

			if (authCred.auth.equals(Authentication.BASIC) || authCred.auth.equals(Authentication.API_KEY)) {
				String credentials = authCred.user + ":" + authCred.password;
				String encodedCredentials = Base64.getEncoder().encodeToString(credentials.getBytes());
				Header[] defaultHeaders = new Header[] {
						new BasicHeader("Authorization", authCred.auth.toString() + " " + encodedCredentials) };
				restClientBuilder.setDefaultHeaders(defaultHeaders);
			} else {
				httpAsyncClientBuilder.setDefaultCredentialsProvider(new BasicCredentialsProvider());
			}
			restClientBuilder.setHttpClientConfigCallback(builder -> httpAsyncClientBuilder);
			restClient = restClientBuilder.build();
		} catch (ConfigException e) {
			restClient = RestClient.builder(httpHosts).build();
			logger.error("[es] Elasticsearch client configuration is invalid, using defaults: {}", e.getMessage());
		}

		ObjectMapper objectMapper = new ObjectMapper();
		objectMapper.registerModule(new AfterburnerModule().setUseValueClassLoader(false));
		JacksonJsonpMapper jsonpMapper = new JacksonJsonpMapper(objectMapper);

		ElasticsearchTransport transport = RetryingRestClientTransport.create(restClient, jsonpMapper,
				ElasticsearchClientConfig.get());
		if (logger.isInfoEnabled()) {
			logger.info("[es] Created client with {} nodes:{}", hosts.size(), hosts.stream().collect(joining(" ")));
		}
		return transport;
	}

	public static void putMeta(String index, String k, String v) throws ElasticIndexException {
		PutMappingResponse response;
		try {
			response = getClient().indices().putMapping(m -> m.index(index).meta(k, JsonData.of(v)));
			logger.info("[es] putMeta({}, {}, {}) => {}", index, k, v, response.acknowledged());
		} catch (ElasticsearchException | IOException e) {
			throw new ElasticIndexException(index, e);
		}
	}

	public static void refreshIndex(String index) {
		if (refreshLocks.computeIfAbsent(index, k -> new ReentrantLock()).tryLock()) {
			try {
				refresh(index);
			} finally {
				refreshLocks.get(index).unlock();
			}
		} else {
			try {
				boolean acquiredLock = refreshLocks.get(index).tryLock(10, TimeUnit.SECONDS);
				if (acquiredLock) {
					refreshLocks.get(index).unlock();
				}
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
			}
		}
	}

	private static void refresh(String index) {
		long time = System.currentTimeMillis();
		try {
			getClient().indices().refresh(r -> r.index(index));
		} catch (ElasticsearchException | IOException e) {
			throw new ElasticIndexException(index, e);
		}
		long ms = (System.currentTimeMillis() - time);
		if (ms > 5) {
			logger.info("time to refresh {}: {}ms", index, ms);
		}
	}

	public static void resetIndexes() {
		indexes.keySet().forEach(ESearchActivator::resetIndex);
	}

	public static void resetIndex(String index) {
		waitForElasticsearchHosts();
		logger.info("Resetting index {}", index);
		ElasticsearchClient esClient = ESearchActivator.getClient();
		deleteIndex(esClient, index);

		if (index.equals("mailspool") && IndexAliasMode.getMode() == Mode.ONE_TO_ONE) {
			activateRingMode();
		}

		initIndex(esClient, index);
	}

	private static void activateRingMode() {
		if (Boolean.getBoolean("es.ring.do.not.force")) {
			return;
		}
		File esRing = new File("/etc/bm/elasticsearch.conf");
		if (!esRing.exists()) {
			try {
				Files.createDirectories(Paths.get("/etc/bm"));
				Files.write(esRing.toPath(), "elasticsearch.indexation.alias_mode.ring = true\n".getBytes());
			} catch (IOException e) {
				throw new ServerFault("error setting up es ring for new installation: " + e.getMessage(), e);
			}
		}
		ElasticsearchClientConfig.reload();
	}

	public static void deleteIndex(String index) {
		waitForElasticsearchHosts();
		ElasticsearchClient esClient = ESearchActivator.getClient();

		logger.info("Deleting index {}", index);
		deleteIndex(esClient, index);
	}

	private static void deleteIndex(ElasticsearchClient esClient, String index) {
		deleteIfExists(esClient, index);
		IndexDefinition indexDefinition = indexes.get(index);
		if (indexDefinition != null) {
			int count = indexDefinition.count();
			boolean isRewritable = indexDefinition.isRewritable();
			if (count > 1 || isRewritable) {
				long realCount = indexNames(esClient).stream() //
						.filter(indexDefinition::supportsIndex) //
						.map(indexName -> deleteIfExists(esClient, indexName)) //
						.count();
				if (count != realCount) {
					logger.warn("Found {} {} indexes which differs from the expected count of {}", realCount, index,
							count);
				}
			}
		}
		logger.info("All matching indices deleted for {}", index);
	}

	private static boolean deleteIfExists(ElasticsearchClient esClient, String index) {
		try {
			esClient.indices().delete(d -> d.index(index));
			logger.info("index '{}' deleted.", index);
			return true;
		} catch (ElasticsearchException e) {
			if (e.error() != null && "index_not_found_exception".equals(e.error().type())) {
				logger.warn("index '{}' not found, can't be delete", index);
				return false;
			}
			throw new ElasticIndexException(index, e);
		} catch (IOException e) {
			throw new ElasticIndexException(index, e);
		}
	}

	public static Optional<String> initIndexIfNotExists(String index) {
		ElasticsearchClient esClient = getClient();
		return indexDefinitionOf(index).map(indexDefinition -> indexNames(esClient).stream() //
				.filter(indexDefinition::supportsIndex) //
				.findFirst() //
				.orElseGet(() -> {
					initIndex(esClient, indexDefinition.index);
					return indexDefinition.index;
				}));
	}

	public static Map<String, String> settingsFromTopology(String indexName) {
		Integer esDataNodes = Math.max(1,
				Topology.getIfAvailable().map(topo -> topo.all(TagDescriptor.bm_es_data.getTag()).size()).orElse(1));
		int replicationFactors = esDataNodes / 2 + 1;
		int numberOfReplicas = replicationFactors - 1;

		if (indexName.equals("mailspool_pending")) {
			numberOfReplicas = 0;
		}
		return Map.of( //
				"settings.index.number_of_replicas", String.valueOf(numberOfReplicas), //
				"settings.index.number_of_shards", String.valueOf(esDataNodes) //
		);
	}

	public static void initIndex(ElasticsearchClient esClient, String index) {
		logger.info("Initialising indices using mode {}", IndexAliasMode.getMode());
		indexDefinitionOf(index).ifPresentOrElse(definition -> {
			IndexAliasCreator mailspoolCreator = IndexAliasCreator.get(definition);
			int count = definition.index.equals(index) ? definition.count() : 1;
			byte[] schema = ESearchActivator.getIndexSchema(definition.index, definition.schema);
			try {
				for (int i = 1; i <= count; i++) {
					String indexName = mailspoolCreator.getIndexName(index, count, i);
					logger.info("init index '{}' with settings & schema", indexName);
					esClient.indices().create(c -> c.index(indexName).withJson(new ByteArrayInputStream(schema)));
					logger.info("index '{}' created, waiting for green...", indexName);
					esClient.cluster().health(h -> h.index(indexName).waitForStatus(HealthStatus.Green));
					definition.rewritableIndex().ifPresent(
							rewritableIndex -> addRewritableIndexAliases(esClient, indexName, rewritableIndex));
					mailspoolCreator.addAliases(index, indexName, definition.count());
					logger.info("added index '{}' aliases", indexName);
				}
			} catch (Exception e) {
				throw new ElasticIndexException(index, e);
			}
		}, () -> {
			logger.warn("no SCHEMA for {}", index);
			try {
				esClient.indices().create(c -> c.index(index));
			} catch (Exception e) {
				throw new ElasticIndexException(index, e);
			}
		});
	}

	public static List<String> indexNames(ElasticsearchClient esClient) {
		try {
			return esClient.indices().resolveIndex(r -> r.name("*")).indices() //
					.stream().map(ResolveIndexItem::name).toList();
		} catch (ElasticsearchException | IOException e) {
			logger.error("[es][indices] Failed to list indices", e);
			return Collections.emptyList();
		}
	}

	private static void addRewritableIndexAliases(ElasticsearchClient esClient, String name, RewritableIndex index) {
		try {
			esClient.indices().updateAliases(u -> u //
					.actions(a -> a.add(add -> add.index(name).alias(index.readAlias()).isWriteIndex(false)))
					.actions(a -> a.add(add -> add.index(name).alias(index.writeAlias()).isWriteIndex(true))));
		} catch (Exception e) {
			throw new ElasticIndexException(name, e);
		}
	}

	private static Optional<IndexDefinition> indexDefinitionOf(String index) {
		return indexes.values().stream().filter(item -> item.supportsIndex(index)).findFirst();
	}

	private static byte[] modifyIndexSchema(byte[] originalSchema, Map<String, String> overrides) {
		ObjectMapper mapper = new ObjectMapper();
		try {
			JsonNode rootNode = mapper.readTree(originalSchema);
			overrides.forEach((key, value) -> {
				String[] path = key.split("\\.");
				JsonNode currentNode = rootNode;
				for (int i = 0; i < path.length - 1; i++) {
					if (currentNode.has(path[i])) {
						currentNode = currentNode.get(path[i]);
					} else if (currentNode instanceof ObjectNode objectNode) {
						currentNode = objectNode.putObject(path[i]);
					} else {
						throw new IllegalStateException("Cannot navigate to " + key);
					}
				}
				if (currentNode instanceof ObjectNode objectNode) {
					objectNode.put(path[path.length - 1], value);
				} else {
					throw new IllegalStateException("Cannot set value for " + key);
				}
			});
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			mapper.writeValue(baos, rootNode);
			return baos.toByteArray();
		} catch (IOException e) {
			throw new IllegalStateException("IOException while writing to byte array output stream", e);
		}
	}

	public static byte[] getIndexSchema(String indexName) {
		return modifyIndexSchema(indexes.get(indexName).schema, settingsFromTopology(indexName));
	}

	public static byte[] getIndexSchema(String indexName, byte[] schema) {
		return modifyIndexSchema(schema, settingsFromTopology(indexName));
	}

	public static int getIndexCount(String index) {
		return indexDefinitionOf(index).map(IndexDefinition::count).orElse(0);
	}

	public static RewritableIndex getRewritableIndex(String indexName) {
		return indexes.get(indexName).rewritableIndex;
	}

	public static void clearClientCache() {
		transports.clear();
	}

	public static MailspoolStats mailspoolStats() {
		return new MailspoolStats(getClient());
	}

	static class IndexDefinition {
		final String index;
		final byte[] schema;
		final ISchemaMatcher matcher;
		final int cnt;
		final RewritableIndex rewritableIndex;
		final boolean supportsAliasRing;

		IndexDefinition(String index, byte[] schema, ISchemaMatcher matcher, int count, boolean rewritable,
				boolean supportsAliasRing) {
			this.index = index;
			this.schema = schema;
			this.matcher = matcher;
			this.cnt = count;
			this.rewritableIndex = rewritable ? RewritableIndex.fromPrefix(index) : null;
			this.supportsAliasRing = supportsAliasRing;
		}

		public int count() {
			return Integer.parseInt(System.getProperty("es." + index + ".count", "" + cnt));
		}

		public boolean isRewritable() {
			return rewritableIndex != null;
		}

		public Optional<RewritableIndex> rewritableIndex() {
			return Optional.ofNullable(rewritableIndex);
		}

		boolean supportsIndex(String name) {
			if (name.equals(index)) {
				return true;
			}
			if (matcher != null) {
				return matcher.supportsIndexName(name);
			}
			if (isRewritable()) {
				return name.startsWith(index + "_");
			}
			return false;
		}
	}

	public static void waitForElasticsearchHosts() {
		Collection<String> hosts = hosts(ES_TAG);
		if (hosts != null) {
			for (String host : hosts) {
				new NetworkHelper(host).waitForListeningPort(9200, 30, TimeUnit.SECONDS);
			}
		}
	}

	public enum Authentication {
		BASIC("Basic"), API_KEY("ApiKey"), NONE("none");

		private String mode;

		Authentication(String m) {
			this.mode = m;
		}

		public String toString() {
			return mode;
		}
	}

	public record AuthenticationCredential(Authentication auth, String user, String password) {

		public AuthenticationCredential(Authentication auth) {
			this(auth, null, null);
		}
	}

	private static String hash(List<String> hosts, AuthenticationCredential authCred) {
		HashFunction hf = Hashing.md5();
		String compactHosts = hosts.stream().reduce("", (f, n) -> f + n);
		HashCode hashCode = hf
				.hashBytes((compactHosts + authCred.user + authCred.password + authCred.auth.toString()).getBytes());
		return hashCode.toString();
	}

	private static Set<CompletableFuture<?>> inFlightRequests = ConcurrentHashMap.newKeySet(128);

	public static <T> CompletableFuture<T> addInFlightAsyncRequest(CompletableFuture<T> ir) {
		var cf = ir.whenComplete((x, t) -> inFlightRequests.remove(ir));
		inFlightRequests.add(ir);
		return cf;
	}

	@VisibleForTesting
	public static void waitAllInFlightAsyncRequests() {
		waitAllInFlightAsyncRequest(30, TimeUnit.SECONDS);
	}

	@VisibleForTesting
	public static void waitAllInFlightAsyncRequest(long timeout, TimeUnit unit) {
		if (inFlightRequests.isEmpty()) {
			return;
		}
		logger.info("Waiting for ElasticSearch inflight request to be finished: " + inFlightRequests);
		try {
			CompletableFuture.allOf(inFlightRequests.toArray(new CompletableFuture[0])).get(timeout, unit);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			if (e instanceof InterruptedException) {
				Thread.currentThread().interrupt();
			}
			logger.error("Unable to wait for all inflight asynchronous requests: {}", e.getMessage());
		}
	}

}
