/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */

package net.bluemind.maintenance.checker.orphan;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import io.netty.util.concurrent.DefaultThreadFactory;
import net.bluemind.backend.mail.replica.api.IInternalMailboxesByLocation;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.maintenance.checker.report.IReport;
import net.bluemind.maintenance.checker.report.IReportLeaf;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.IInternalContainersByLocation;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.mailbox.api.Mailbox;

public class OrphanChecker {
	private final boolean autoRepair;
	private final boolean forceDeleteManyOrphans;
	private final String datalocation;
	private final int nbWorkers;
	private final IServiceProvider serviceProvider;
	private final IServiceProvider serviceProviderInfiniteQuery;

	public OrphanChecker(IServiceProvider serviceProvider, IServiceProvider serviceProviderInfiniteQuery,
			String datalocation, boolean autoRepair, int nbWorkers, boolean forceDeleteManyOrphans) {
		this.serviceProvider = serviceProvider;
		this.serviceProviderInfiniteQuery = serviceProviderInfiniteQuery;
		this.autoRepair = autoRepair;
		this.forceDeleteManyOrphans = forceDeleteManyOrphans;
		this.datalocation = datalocation;
		this.nbWorkers = nbWorkers;
	}

	public void check(IReport report, String[] containerTypes, List<String> domainUids) {
		Arrays.asList(containerTypes).forEach(type -> check(type, report, domainUids));
	}

	private void check(String containerType, IReport report, List<String> domainUids) {
		IInternalContainersByLocation icontainersService = serviceProvider.instance(IInternalContainersByLocation.class,
				datalocation);
		ContainerQuery containerQuery = ContainerQuery.type(containerType);
		List<BaseContainerDescriptor> containersByType = icontainersService.listByType(containerQuery);

		Map<String, List<BaseContainerDescriptor>> containerByDomain = containersByType.stream().collect(Collectors
				.groupingBy(BaseContainerDescriptor::getDomainUid, Collectors.mapping(c -> c, Collectors.toList())));

		ArrayBlockingQueue<String> submitQueue = new ArrayBlockingQueue<>(nbWorkers);
		try (ExecutorService pool = Executors.newFixedThreadPool(nbWorkers,
				new DefaultThreadFactory("orphan-containers-check"))) {

			for (String domainUid : domainUids) {
				try {
					submitQueue.put(domainUid);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					report.info("Interrupted");
					return;
				}
				pool.submit(() -> {
					try {
						checkOrphanByDomain(report.addDomainReport(domainUid), containerType,
								containerByDomain.get(domainUid), domainUid);
					} catch (Exception e) {
						report.error("unable to check: {}", e.getMessage());
					} finally {
						submitQueue.remove(domainUid); // NOSONAR: don't care
					}
				});
			}
			pool.shutdown();
			pool.awaitTermination(5, TimeUnit.DAYS);
		} catch (InterruptedException e) {
			report.warn("Interrupted");
			Thread.currentThread().interrupt();
		}
	}

	private void checkOrphanByDomain(IReportLeaf domainReport, String containerType,
			List<BaseContainerDescriptor> containers, String domainUid) {
		domainReport.info("Check owner exists for '{}' containers in domain {}", containerType, domainUid);

		IDirectory dirApi = serviceProvider.instance(IDirectory.class, domainUid);
		DirEntryQuery q = DirEntryQuery.filterKind(Kind.USER, Kind.GROUP, Kind.MAILSHARE, Kind.RESOURCE);
		q.systemFilter = false;
		q.hiddenFilter = false;

		List<String> dirEntries = dirApi.search(q).values.stream().map(dv -> dv.value.entryUid).toList();
		if (dirEntries.isEmpty()) {
			domainReport.info("No directory entry found, do nothing.");
			return;
		}

		domainReport.info("Dir entries {}", dirEntries);

		Map<String, List<BaseContainerDescriptor>> containerByOwners = containers.stream()
				.filter(c -> !dirEntries.contains(c.owner)).collect(Collectors.groupingBy(
						BaseContainerDescriptor::getOwner, Collectors.mapping(c -> c, Collectors.toList())));

		domainReport.info("Orphans {}",
				containerByOwners.entrySet().stream().map(
						e -> e.getKey() + " : " + e.getValue().stream().map(c -> c.uid + "(" + c.owner + ")").toList())
						.toList());

		if (containerByOwners.keySet().size() > dirEntries.size() * 0.33) {
			if (forceDeleteManyOrphans) {
				domainReport.warn("Orphan cleanup would remove more than 33% of email data");
			} else {
				domainReport.error(
						"Orphan cleanup would remove more than 33% of email data, I refuse to do that (re-run with option --force-delete-many-orphans to do it)");
				return;
			}
		}

		IContainers containersService = serviceProvider.instance(IContainers.class);

		for (Entry<String, List<BaseContainerDescriptor>> containersEntry : containerByOwners.entrySet()) {
			int deleteCount = containersEntry.getValue().size();
			for (BaseContainerDescriptor container : containersEntry.getValue()) {
				try {
					domainReport.error("Missing owner {} for container {}({}) on domain {}", container.owner,
							container.uid, container.name, domainUid);
					if (autoRepair) {
						domainReport.info("Delete records...");
						serviceProviderInfiniteQuery.instance(IInternalMailboxesByLocation.class, datalocation)
								.deleteMailbox(container.uid);
						domainReport.info("Delete container...");
						containersService.delete(container.uid);
						domainReport.info("Delete complete for container {}", container.uid);
						deleteCount--;
					} else {
						domainReport.warn("Re-run command without --no-repair to execute orphan records deletion");
					}
				} catch (Exception e) {
					domainReport.error("Error trying to delete container {}({}) owner {}: {}", container.uid,
							container.name, container.owner, e.getMessage());
				}
			}

			if (autoRepair) {
				String subtreeUid = IMailReplicaUids.subtreeUid(domainUid, Mailbox.Type.user, containersEntry.getKey());
				if (deleteCount > 0) {
					domainReport.error("Cannot delete subtree container {}: because child {} containers still exist",
							subtreeUid, deleteCount);
				} else {
					try {
						containersService.delete(subtreeUid);
						domainReport.info("Delete complete for owner {}", containersEntry.getKey());
					} catch (Exception e) {
						domainReport.error("Error trying to delete subtree container {}: {}", subtreeUid,
								e.getMessage());
					}
				}
			}
		}
	}
}
