/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.authentication.service;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.UUID;

import com.google.common.hash.Hashing;

import net.bluemind.authentication.api.APIKey;
import net.bluemind.authentication.api.IAPIKeys;
import net.bluemind.authentication.repository.IAPIKeyStore;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.role.api.BasicRoles;
import net.bluemind.tx.outbox.api.ITxOutbox;

public class APIKeysService implements IAPIKeys {

	private static final String INVALID_SECURITY_CONTEXT = "Invalid securityContext";
	private final IAPIKeyStore store;
	private final SecurityContext context;
	private final RBACManager rbac;
	private ITxOutbox outbox;

	public APIKeysService(BmContext bmCtx) {
		store = RepositoryProvider.instance(IAPIKeyStore.class, bmCtx);
		this.context = bmCtx.getSecurityContext();
		this.rbac = new RBACManager(bmCtx);
		this.outbox = bmCtx.su().provider().instance(ITxOutbox.class, "global.virt", "system", "apikeys", "apikeys",
				DataLocation.directory().serverUid());
	}

	@Override
	public APIKey store(String key, String displayName) throws ServerFault {
		if (context.isAnonymous()) {
			throw new ServerFault(INVALID_SECURITY_CONTEXT, ErrorCode.PERMISSION_DENIED);
		}

		if (displayName == null || displayName.trim().isEmpty()) {
			throw new ServerFault("API key display name cannot be empty");
		}

		APIKey apikey = new APIKey();
		apikey.sid = key;
		apikey.displayName = displayName;
		apikey.subject = context.getSubject();
		apikey.domainUid = context.getContainerUid();
		TxEnabler.atomically(() -> {
			store.create(apikey);
			outbox.forKafka(ItemValue.create(item(apikey.sid), apikey), null, false);
		});
		return store.get(apikey.sid);
	}

	private Item item(String sid) {
		String kUid = context.getSubject() + "-" + sid;
		return Item.create(kUid, Hashing.sipHash24().hashString(kUid, StandardCharsets.US_ASCII).asLong(),
				ItemFlag.Seen);
	}

	@Override
	public APIKey create(String displayName) throws ServerFault {
		return store(UUID.randomUUID().toString(), displayName);
	}

	@Override
	public void delete(String sid) throws ServerFault {
		if (context.isAnonymous()) {
			throw new ServerFault(INVALID_SECURITY_CONTEXT, ErrorCode.PERMISSION_DENIED);
		}

		TxEnabler.atomically(() -> {
			store.delete(sid);
			outbox.forKafka(ItemValue.create(item(sid), new APIKey()), null, true);
		});
	}

	@Override
	public List<APIKey> list() throws ServerFault {
		if (context.isAnonymous()) {
			throw new ServerFault(INVALID_SECURITY_CONTEXT, ErrorCode.PERMISSION_DENIED);
		}

		return store.list();
	}

	@Override
	public List<APIKey> listAll() throws ServerFault {
		rbac.check(BasicRoles.ROLE_ADMIN);

		return store.listAll();
	}

	@Override
	public APIKey get(String sid) throws ServerFault {
		return store.get(sid);
	}

}
