/* eslint-disable @typescript-eslint/no-non-null-assertion */
import pRetry from "p-retry";
import { FolderCounters, MailboxFolder, MailboxFoldersClient } from "@bluemind/backend.mail.api";
import { ItemFlag, ItemValue } from "@bluemind/core.container.api";
import logger from "@bluemind/logger";
import db from "../MailDB";
import session from "@bluemind/session";

export default class extends MailboxFoldersClient {
    next?: (...args: Array<unknown>) => Promise<never>;
    async all(): Promise<ItemValue<MailboxFolder>[]> {
        try {
            const uid = `${this.mailboxRoot}@${this.partition}`;
            return await retry(async () => {
                if (await db.isSubscribed(uid)) {
                    return db.getAllMailFolders(this.mailboxRoot);
                }
                return this.next!();
            });
        } catch (error) {
            console.debug(error);
        }
        return this.next!();
    }
    async counters(folderItemIds?: Array<number>): Promise<Array<FolderCounters>> {
        try {
            const uid = `${this.mailboxRoot}@${this.partition}`;
            if (await db.isSubscribed(uid)) {
                const ids = new Set(folderItemIds || []);
                const folders = await db.getAllMailFolders(this.mailboxRoot);
                const counters = await Promise.all(
                    folders
                        .filter(({ internalId }) => ids.has(internalId!))
                        .map(async folder => {
                            const items = await db.getAllMailItemLight(folder.uid!);
                            const unseenVisible = items.filter(item => !item.flags.includes(ItemFlag.Seen)).length;
                            const total = items.length;
                            return {
                                itemId: folder.internalId,
                                unseenVisible,
                                totalVisible: total,
                                total
                            };
                        })
                );
                return counters;
            }
        } catch (error) {
            console.debug(error);
        }
        return this.next!();
    }
}

async function retry<T>(fn: () => Promise<T>): Promise<T> {
    const wrapToThrowErrorOnFailure = <T>(fnToWrap: () => Promise<T>): (() => Promise<T>) => {
        return () =>
            fnToWrap().catch((error: any) => {
                logger.log("catching an error", error);
                throw new Error(error);
            });
    };
    return pRetry(wrapToThrowErrorOnFailure(fn), { retries: 1, onFailedAttempt: () => session.revalidate() });
}
