import { openDB, DBSchema, IDBPDatabase, IDBPTransaction, StoreNames, StoreValue } from "idb";
import { sortedIndexBy } from "@bluemind/commons/utils/array";

import { ContainerSubscriptionModel, ItemFlag, ItemValue } from "@bluemind/core.container.api";
import { MailboxFolder, MailboxItem } from "@bluemind/backend.mail.api";
import session from "@bluemind/session";

import logger from "@bluemind/logger";

export type SyncOptionsType = "mail_folder" | "mail_item" | "owner_subscriptions";

export interface SyncOptions {
    uid: string;
    version: number;
    type: SyncOptionsType;
    pending?: boolean;
}

export type MailItemLight = {
    internalId: number;
    flags: ItemFlag[];
    date: number;
    subject?: string;
    size: number;
    sender?: string;
};

interface Reconciliation<T> {
    uid: string;
    items: T[];
    deletedIds: number[];
}
interface MailSchema extends DBSchema {
    mail_folders: {
        key: string;
        value: ItemValue<MailboxFolder>;
        indexes: { "by-mailboxRoot": string };
    };
    sync_options: {
        key: string;
        value: SyncOptions;
        indexes: { "by-type": string };
    };
    mail_items: {
        key: [string, number];
        value: ItemValue<MailboxItem>;
        indexes: { "by-folderUid": string };
    };
    mail_item_light: {
        key: string;
        value: MailItemLight[];
    };
    owner_subscriptions: {
        key: string;
        value: ItemValue<ContainerSubscriptionModel>;
        indexes: { "by-type": string };
    };
}

type MailDBTransaction = IDBPTransaction<MailSchema, StoreNames<MailSchema>[], IDBTransactionMode>;

export interface MailDB {
    getSyncOptions(uid: string): Promise<SyncOptions | undefined>;
    updateSyncOptions(options: SyncOptions): Promise<string | undefined>;
    isSubscribed(uid: string): Promise<boolean>;
    deleteSyncOptions(uid: string): Promise<void>;
    deleteOwnerSubscriptions(deletedIds: number[]): Promise<void>;
    deleteMailFolders(mailboxRoot: string, deletedIds: number[]): Promise<void>;
    putMailFolders(mailboxRoot: string, items: ItemValue<MailboxFolder>[], tx?: MailDBTransaction): Promise<void>;
    putOwnerSubscriptions(items: ItemValue<ContainerSubscriptionModel>[], tx?: MailDBTransaction): Promise<void>;
    resetMailFolder(folderUid: string): Promise<void>;
    resetMailbox(mailboxRoot: string): Promise<void>;
    getAllMailItemLight(folderUid: string, tx?: MailDBTransaction): Promise<MailItemLight[]>;
    getMailItems(folderUid: string, ids: number[]): Promise<(ItemValue<MailboxItem> | undefined)[]>;
    getAllMailFolders(mailboxRoot: string): Promise<ItemValue<MailboxFolder>[]>;
    getOwnerSubscriptions(type: string): Promise<ItemValue<ContainerSubscriptionModel>[]>;
    getAllOwnerSubscriptions(): Promise<ItemValue<ContainerSubscriptionModel>[]>;
    reconciliate(data: Reconciliation<ItemValue<MailboxItem>>, syncOptions: SyncOptions): Promise<void>;
    setMailItemLight(
        folderUid: string,
        items: ItemValue<MailboxItem>[],
        deleted: number[],
        tx?: MailDBTransaction
    ): Promise<void>;
    consolidateHierarchy(mailboxRoot: string): Promise<void>;
}

export class MailDBImpl implements MailDB {
    dbPromise: Promise<IDBPDatabase<MailSchema>>;
    constructor(mailboxRoot: string) {
        this.dbPromise = this.openDB(mailboxRoot);
    }

    private async openDB(mailboxRoot: string): Promise<IDBPDatabase<MailSchema>> {
        const schemaVersion = 11;
        return openDB<MailSchema>(`${mailboxRoot}:webapp/mail`, schemaVersion, {
            upgrade(db, oldVersion) {
                logger.log(`[SW][DB] Upgrading from ${oldVersion} to ${schemaVersion}`);
                if (oldVersion < schemaVersion) {
                    logger.log("[SW][DB] Upgrading deleting existing object store");
                    for (const name of Object.values(db.objectStoreNames)) {
                        db.deleteObjectStore(name);
                    }
                }
                db.createObjectStore("sync_options", { keyPath: "uid" }).createIndex("by-type", "type");
                db.createObjectStore("mail_items", { keyPath: ["folderUid", "internalId"] }).createIndex(
                    "by-folderUid",
                    "folderUid"
                );
                db.createObjectStore("mail_folders", { keyPath: "uid" }).createIndex("by-mailboxRoot", "mailboxRoot");
                db.createObjectStore("mail_item_light");
                db.createObjectStore("owner_subscriptions", { keyPath: "uid" }).createIndex(
                    "by-type",
                    "value.containerType"
                );
            },
            blocking: async () => {
                await this.close();
                this.dbPromise = this.openDB(mailboxRoot);
            }
        });
    }

    private async getTx<StoreName extends StoreNames<MailSchema>>(
        storeName: StoreName,
        mode: IDBTransactionMode,
        tx?: IDBPTransaction<MailSchema, StoreName[], IDBTransactionMode>
    ): Promise<IDBPTransaction<MailSchema, StoreName[], IDBTransactionMode>> {
        return tx || (await this.dbPromise).transaction([storeName], mode);
    }

    private async putItems<T extends StoreValue<MailSchema, StoreName>, StoreName extends StoreNames<MailSchema>>(
        items: T[],
        storeName: StoreName,
        optionalTransaction?: IDBPTransaction<MailSchema, StoreName[], IDBTransactionMode>
    ) {
        const tx = optionalTransaction || (await this.dbPromise).transaction(storeName, "readwrite");
        await Promise.all(items.map(item => tx.objectStore(storeName).put?.(item)));
        await tx.done;
    }

    private async putMailItems(items: ItemValue<MailboxItem>[], tx?: MailDBTransaction) {
        await this.putItems(items, "mail_items", tx);
    }
    async close(): Promise<void> {
        (await this.dbPromise).close();
    }
    async getSyncOptions(uid: string) {
        return (await this.dbPromise).get("sync_options", uid);
    }

    async updateSyncOptions(syncOptions: SyncOptions) {
        const actual = await this.getSyncOptions(syncOptions.uid);
        if (
            actual === undefined ||
            actual.version < syncOptions.version ||
            (actual.version === syncOptions.version && actual.pending !== syncOptions.pending)
        ) {
            return (await this.dbPromise).put("sync_options", syncOptions);
        }
    }

    async isSubscribed(uid: string) {
        const key = await (await this.dbPromise).getKey("sync_options", uid);
        return key !== undefined;
    }

    async deleteSyncOptions(uid: string) {
        await (await this.dbPromise).delete("sync_options", uid);
    }

    async deleteOwnerSubscriptions(deletedIds: number[]) {
        const subscriptionUidsToDelete = (await this.getAllOwnerSubscriptions())
            .filter(ownerSubscription => deletedIds.includes(ownerSubscription.internalId as number))
            .map(ownerSubscription => ownerSubscription.uid);
        const tx = (await this.dbPromise).transaction("owner_subscriptions", "readwrite");
        tx.onerror = event => {
            logger.error("[SW][DB] Failed to delete owner subscriptions", deletedIds, event);
        };
        subscriptionUidsToDelete.forEach(uid => tx.objectStore("owner_subscriptions").delete(uid as string));
        await tx.done;
    }

    async deleteMailFolders(mailboxRoot: string, deletedIds: number[]) {
        const uids = (await this.getAllMailFolders(mailboxRoot))
            .filter(({ internalId }) => deletedIds.includes(internalId as number))
            .map(({ uid }) => uid as string);

        await this.deleteMailboxFolderByUid(uids);
    }

    private async deleteMailboxFolderByUid(uids: string[]) {
        const tx = (await this.dbPromise).transaction("mail_folders", "readwrite");
        uids.forEach(uid => tx.objectStore("mail_folders").delete(uid));
        await tx.done;
    }

    async putMailFolders(mailboxRoot: string, items: ItemValue<MailboxFolder>[], tx?: MailDBTransaction) {
        await this.putItems(
            items.map(item => ({ ...item, mailboxRoot })),
            "mail_folders",
            tx
        );
    }

    async putOwnerSubscriptions(items: ItemValue<ContainerSubscriptionModel>[], tx?: MailDBTransaction) {
        await this.putItems(items, "owner_subscriptions", tx);
    }

    async getAllMailItemLight(folderUid: string, tx?: MailDBTransaction): Promise<MailItemLight[]> {
        const _tx = await this.getTx("mail_item_light", "readonly", tx);
        const store = _tx.objectStore("mail_item_light");
        return (await store.get(folderUid)) || [];
    }

    async getMailItems(folderUid: string, ids: number[]) {
        const tx = (await this.dbPromise).transaction(["mail_items"], "readonly");
        return Promise.all(ids.map(id => tx.objectStore("mail_items").get([folderUid, id])));
    }

    async getAllMailFolders(mailboxRoot: string) {
        return (await this.dbPromise).getAllFromIndex("mail_folders", "by-mailboxRoot", mailboxRoot);
    }

    async getOwnerSubscriptions(type: string) {
        return (await this.dbPromise).getAllFromIndex("owner_subscriptions", "by-type", type);
    }

    async getAllOwnerSubscriptions() {
        return (await this.dbPromise).getAll("owner_subscriptions");
    }

    async reconciliate(data: Reconciliation<ItemValue<MailboxItem>>, syncOptions: SyncOptions) {
        const { items, uid, deletedIds } = data;
        const tx = (await this.dbPromise).transaction(["sync_options", "mail_items", "mail_item_light"], "readwrite");
        this.putMailItems(
            items.map(mailItem => ({ ...mailItem, folderUid: uid })),
            tx
        );
        this.setMailItemLight(uid, items, deletedIds, tx);
        deletedIds.map(id => tx.objectStore("mail_items").delete([uid, id]));
        await tx.objectStore("sync_options").put(syncOptions);
        await tx.done;
    }
    async setMailItemLight(
        folderUid: string,
        items: ItemValue<MailboxItem>[],
        deleted: number[],
        optTx?: IDBPTransaction<MailSchema, StoreNames<MailSchema>[], IDBTransactionMode>
    ) {
        const lights: Array<MailItemLight> = await this.getAllMailItemLight(folderUid, optTx);
        deleted.forEach(id => {
            const dummy = { internalId: id, flags: [], date: 0, subject: "", size: 0, sender: "" };
            const index = sortedIndexBy(lights, dummy, "internalId");
            if (lights[index]?.internalId === id) {
                lights.splice(index, 1);
            }
        });
        items.forEach(mail => {
            const light = toLight(mail);
            const index = sortedIndexBy(lights, light, "internalId");
            if (lights[index]?.internalId === light.internalId) {
                lights.splice(index, 1, light);
            } else {
                lights.splice(index, 0, light);
            }
        });
        const tx = (await this.dbPromise).transaction("mail_item_light", "readwrite");

        await tx.objectStore("mail_item_light").put(lights, folderUid);
        await tx.done;
    }

    async consolidateHierarchy(mailboxRoot: string) {
        const folders = await this.getAllMailFolders(mailboxRoot);

        const deprecatedFoldersUids = getDeprecatedFoldersUids(folders);

        await this.deleteMailboxFolderByUid(deprecatedFoldersUids);
        logger.log("[SW][DB] DB consolidated: " + deprecatedFoldersUids.length + " folder(s) removed");
    }

    async resetMailFolder(folderUid: string): Promise<void> {
        const tx = (await this.dbPromise).transaction(["mail_item_light", "mail_items"], "readwrite");
        const items = await this.getAllMailItemLight(folderUid, tx);
        await Promise.all(items.map(item => tx.objectStore("mail_items").delete([folderUid, item.internalId])));
        await tx.objectStore("mail_item_light").delete(folderUid);
        await tx.done;
    }

    async resetMailbox(mailboxRoot: string): Promise<void> {
        const stores = Array.from((await this.dbPromise).objectStoreNames);

        const tx = (await this.dbPromise).transaction(stores, "readwrite");
        await Promise.all(stores.map(store => tx.objectStore(store).clear()));
        await tx.done;
    }
}

function getDeprecatedFoldersUids(folders: ItemValue<MailboxFolder>[]) {
    const parentWithChildren = new Set();
    const remove = new Set();
    folders.sort(byFullnameInDescOrder).forEach(aFolder => {
        if (isDeprecated(aFolder)) {
            remove.add(aFolder.uid);
        } else {
            parentWithChildren.add(aFolder.value.parentUid);
        }
    });
    return Array.from(remove).filter((uid): uid is string => uid != null);

    function byFullnameInDescOrder(a: ItemValue<MailboxFolder>, b: ItemValue<MailboxFolder>) {
        return (b.value.fullName || "")?.localeCompare(a.value.fullName || "");
    }
    function isDeprecated(aFolder: ItemValue<MailboxFolder>) {
        return !parentWithChildren.has(aFolder.uid) && aFolder.value.virtualFolder;
    }
}

function toLight(mail: ItemValue<MailboxItem>): MailItemLight {
    return {
        internalId: mail.internalId as number,
        flags: mail.flags as ItemFlag[],
        date: mail.value.body.date as number,
        subject: mail.value.body.subject?.toLowerCase().replace(/^(\W*|re\s*:)*/i, ""),
        size: mail.value.body.size as number,
        sender: mail.value.body.recipients?.find(recipient => recipient.kind === "Originator")?.address?.toLowerCase()
    };
}

let implementation: MailDBImpl | null = null;

// Export for testing purpose
export async function instance(): Promise<MailDBImpl> {
    if (!implementation) {
        const name = `user.${await session.userId}@${(await session.domain).replace(/\./g, "_")}`;
        implementation = new MailDBImpl(name);
    }
    return implementation;
}

session.addEventListener("change", event => {
    const { old, value } = event.detail;
    if (value.userId != old?.userId && implementation) {
        implementation?.close();
        implementation = null;
    }
});

const db: MailDB = {
    getSyncOptions: uid => instance().then(db => db.getSyncOptions(uid)),
    updateSyncOptions: options => instance().then(db => db.updateSyncOptions(options)),
    isSubscribed: uid => instance().then(db => db.isSubscribed(uid)),
    deleteSyncOptions: uid => instance().then(db => db.deleteSyncOptions(uid)),
    deleteOwnerSubscriptions: deletedIds => instance().then(db => db.deleteOwnerSubscriptions(deletedIds)),
    deleteMailFolders: (mailboxRoot, deletedIds) =>
        instance().then(db => db.deleteMailFolders(mailboxRoot, deletedIds)),
    putMailFolders: (mailboxRoot, items, tx) => instance().then(db => db.putMailFolders(mailboxRoot, items, tx)),
    putOwnerSubscriptions: (items, tx) => instance().then(db => db.putOwnerSubscriptions(items, tx)),
    resetMailFolder: (folderUid: string) => instance().then(db => db.resetMailFolder(folderUid)),
    resetMailbox: (mailboxRoot: string) => instance().then(db => db.resetMailbox(mailboxRoot)),
    getAllMailItemLight: (folderUid, tx) => instance().then(db => db.getAllMailItemLight(folderUid, tx)),
    getMailItems: (folderUid, ids) => instance().then(db => db.getMailItems(folderUid, ids)),
    getAllMailFolders: mailboxRoot => instance().then(db => db.getAllMailFolders(mailboxRoot)),
    getOwnerSubscriptions: type => instance().then(db => db.getOwnerSubscriptions(type)),
    getAllOwnerSubscriptions: () => instance().then(db => db.getAllOwnerSubscriptions()),
    reconciliate: (data, syncOptions) => instance().then(db => db.reconciliate(data, syncOptions)),
    setMailItemLight: (folderUid, items, deleted, tx) =>
        instance().then(db => db.setMailItemLight(folderUid, items, deleted, tx)),
    consolidateHierarchy: mailboxRoot => instance().then(db => db.consolidateHierarchy(mailboxRoot))
};

export default db;
