/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.incremental.indexing.cql.store;

import java.sql.SQLException;
import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.uuid.Uuids;

import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore;

public class CqlIncrementalIndexingStore extends CqlAbstractStore implements IIncrementalIndexingStore {

	public static final int CONTAINER_BUCKETS = 256;

	public CqlIncrementalIndexingStore(CqlSession s) {
		super(s);
	}

	@Override
	public ContainerSyncState getState(ContainerUid cuid) throws SQLException {
		ContainerSyncState state = unique("select version from t_indexed_versions where container_uid=?",
				r -> new ContainerSyncState(cuid, r.getLong(0)), voidPop(), cuid.value());
		if (state == null) {
			state = new ContainerSyncState(cuid, 0L);
		}
		return state;
	}

	@Override
	public void markDirty(ContainerUid c) throws SQLException {
		voidCql("insert into t_dirty_indices (bucket_id, container_uid, dirtied_at) values (?,?,?)", bucketId(c),
				c.value(), Uuids.timeBased());
	}

	private int bucketId(ContainerUid c) {
		return bucketId(c.value());
	}

	private int bucketId(String contUid) {
		return Math.abs(contUid.hashCode() % CONTAINER_BUCKETS);
	}

	@Override
	public boolean checkpointSync(ContainerSyncState stateUpdate, DirtynessMarker marker) throws SQLException {
		voidCql("update t_indexed_versions set version=? where container_uid=?", //
				stateUpdate.version(), stateUpdate.cont().value());
		UUID end = Uuids.endOf(marker.since().toEpochMilli());
		return voidCql("delete from t_dirty_indices where bucket_id=? AND container_uid=? IF dirtied_at <= ?",
				bucketId(marker.container()), marker.container().value(), end);
	}

	@Override
	public List<DirtynessMarker> fetchNexts(Set<String> inprogressids, int limit) throws SQLException {

		Set<Integer> excludeBuckets = inprogressids.stream().map(this::bucketId).collect(Collectors.toSet());

		return IntStream.range(0, CONTAINER_BUCKETS).parallel().mapToObj(b -> b)
				.filter(b -> !excludeBuckets.contains(b))
				.flatMap(bucket -> map(
						"select container_uid, dirtied_at from t_dirty_indices_by_date where bucket_id=? limit ?",
						r -> new DirtynessMarker(ContainerUid.of(r.getString(0)),
								Instant.ofEpochMilli(Uuids.unixTimestamp(r.getUuid(1)))),
						voidPop(), bucket, limit).stream())
				.limit(limit).toList();
	}

	@Override
	public void forgetStates() {
		batch(//
				b("delete from t_dirty_indices"), //
				b("delete from t_indexed_versions")//
		);
	}

	@Override
	public long slowCostlyCount() throws SQLException {
		return IntStream.range(0, CONTAINER_BUCKETS).parallel()
				.mapToLong(bucket -> unique("select count(*) from t_dirty_indices where bucket_id=?", r -> r.getLong(0),
						voidPop(), bucket))
				.sum();
	}

}
