package net.bluemind.tika.server.impl;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Throwables;
import com.google.common.hash.Hashing;
import com.google.common.hash.HashingOutputStream;
import com.google.common.io.CountingOutputStream;

import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import net.bluemind.common.io.Buffered;

public final class BinaryFileDataHandler implements Handler<Buffer> {

	private static final Logger logger = LoggerFactory.getLogger(BinaryFileDataHandler.class);
	private File file;
	private String path;
	private final HashingOutputStream hashing;
	private final CountingOutputStream out;

	@SuppressWarnings("deprecation")
	public BinaryFileDataHandler() {
		try {
			this.file = File.createTempFile("tika", ".bin", TikaDirectories.WORK);
			this.hashing = new HashingOutputStream(Hashing.md5(), Buffered.output(new FileOutputStream(file)));
			this.out = new CountingOutputStream(hashing);
			this.path = file.getAbsolutePath();
		} catch (IOException e) {
			throw Throwables.propagate(e);
		}
	}

	public void cleanup() {
		file.delete();
		logger.info("Cleaned {}bytes at {}", out.getCount(), path);
	}

	public String flushAndHash() {
		try {
			out.close();
			String hash = hashing.hash().toString();
			logger.info("Flushed {}bytes, hash => {}", out.getCount(), hash);
			return hash;
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
			return null;
		}
	}

	@Override
	public void handle(Buffer event) {
		try {
			byte[] bytes = event.getBytes();
			out.write(bytes);
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}
	}

	public String getFilePath() {
		return path;
	}

}
