/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cli.mail;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.google.common.hash.Hashing;

import net.bluemind.backend.cyrus.index.CyrusIndex;
import net.bluemind.backend.cyrus.index.CyrusIndexRecord;
import net.bluemind.backend.cyrus.partitions.CyrusFileSystemPathHelper;
import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.cyrus.partitions.MailboxDescriptor;
import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.IMailboxFoldersByContainer;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.replica.api.AppendTx;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.ImapBinding;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.delivery.conversationreference.api.IConversationReference;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.lib.jutf7.UTF7Converter;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mailbox.api.Mailbox.Type;
import net.bluemind.network.topology.Topology;
import net.bluemind.node.api.FileDescription;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.Server;
import net.bluemind.system.api.ISystemConfiguration;
import picocli.CommandLine.Command;

@Command(name = "resync-cyrus-spool", description = "Resync files in the cyrus spool")
public class ResyncCyrusSpoolCommand implements ICmdLet, Runnable {

	private CliContext ctx;
	private CliUtils cliUtils;

	@Override
	public void run() {
		cliUtils.getDomainUids().forEach(this::syncDomain);
	}

	private void syncDomain(String domain) {

		String sysConfKey = "sds_cyrus_spool_enabled";
		Boolean mailMigrationIsNotYetFinished = ctx.adminApi().instance(ISystemConfiguration.class).getValues()
				.booleanValue(sysConfKey);
		if (mailMigrationIsNotYetFinished != null && mailMigrationIsNotYetFinished.booleanValue()) {
			ctx.error("The mail migration has not yet finished, aborting resync to prevent duplicated mails");
			System.exit(1);
		}

		ctx.info("Synchronizing Cyrus spool of {}", domain);
		IDirectory directoryApi = ctx.adminApi().instance(IDirectory.class, domain);
		var query = DirEntryQuery.filterKind(Kind.USER, Kind.MAILSHARE, Kind.GROUP);
		directoryApi.search(query).values.forEach(dirEntry -> syncMailbox(domain, dirEntry));
	}

	private void syncMailbox(String domainUid, ItemValue<DirEntry> mailbox) {
		String logId = !Strings.isNullOrEmpty(mailbox.value.email)
				? (mailbox.value.email + " (" + mailbox.value.entryUid + ")")
				: mailbox.value.entryUid;
		ctx.info("Synchronizing Cyrus spool of {}", logId);

		ContainerQuery query = ContainerQuery.ownerAndType(mailbox.uid, IMailReplicaUids.MAILBOX_RECORDS);
		long errorCount = migrateCyrusSpool(logId, domainUid, mailbox,
				ctx.adminApi().instance(IContainers.class).all(query));
		if (errorCount != 0) {
			ctx.error("Some errors ({}) encountered while compressiong emails", errorCount);
		}

	}

	private long migrateCyrusSpool(String logId, String domainUid, ItemValue<DirEntry> mailbox,
			List<ContainerDescriptor> containerDescriptors) {
		long errorCount = 0;
		if (containerDescriptors.isEmpty()) {
			return 0;
		}

		IServiceProvider prov = ctx.adminApi();
		IMailboxes mboxes = prov.instance(IMailboxes.class, domainUid);
		ItemValue<Mailbox> mbox = mboxes.getComplete(mailbox.value.entryUid);
		IDbReplicatedMailboxes folderApi = prov.instance(IDbReplicatedMailboxes.class, domainUid,
				mbox.value.type.nsPrefix + mbox.value.name);
		for (ContainerDescriptor containerDesc : containerDescriptors) {
			String folderUid = containerDesc.uid.substring(IMailReplicaUids.MAILBOX_RECORDS_PREFIX.length());
			ItemValue<MailboxFolder> folder = folderApi.getComplete(folderUid);
			if (folder == null) {
				continue;
			}
			String logIdF = logId + ":" + folder.value.fullName;
			IDbMailboxRecords mailboxItems = ctx.adminApi().instance(IDbMailboxRecords.class, folder.uid);
			ItemValue<Server> server = Topology.get().datalocation(containerDesc.datalocation);
			INodeClient nc = NodeActivator.get(server.value.address());
			CyrusPartition part = CyrusPartition.forServerAndDomain(server, domainUid);
			MailboxDescriptor desc = new MailboxDescriptor();
			desc.type = mbox.value.type;
			desc.mailboxName = mbox.value.name;
			desc.utf7FolderPath = UTF7Converter.encode(folder.value.fullName);
			Path inputLiveFolderPath = Paths.get(CyrusFileSystemPathHelper.getFileSystemPath(domainUid, desc, part, 1))
					.getParent();
			Path inputArchiveFolderPath = Paths
					.get(CyrusFileSystemPathHelper.getHSMFileSystemPath(domainUid, desc, part, 1)).getParent();
			List<ImapBinding> errors = new ArrayList<>();
			long expunged = 0;
			AtomicLong remaining = new AtomicLong();
			Path cyrusIndexPath = Paths
					.get(CyrusFileSystemPathHelper.getMetaFileSystemPath(domainUid, desc, part, "cyrus.index"));
			if (nc.exists(cyrusIndexPath.toAbsolutePath().toString())) {
				List<Integer> errorsExpunge = new ArrayList<>();
				try {
					InputStream in = nc.openStream(cyrusIndexPath.toAbsolutePath().toString());
					CyrusIndex index = new CyrusIndex(in);
					for (CyrusIndexRecord rec : index.readAll()) {
						// expunged records
						if (rec.systemFlags[0] == (byte) (1 << 7)) {
							Path inputFilePath = inputLiveFolderPath.resolve(rec.uid + ".");
							if (nc.exists(inputFilePath.toString())) {
								try {
									nc.deleteFile(inputFilePath.toString());
									expunged++;
								} catch (ServerFault e) {
									ctx.error("[{}] Unable to delete file, continue, but with ending error: {}", logId,
											e.getMessage());
									errorsExpunge.add(rec.uid);
								}
							}
							inputFilePath = inputArchiveFolderPath.resolve(rec.uid + ".");
							if (nc.exists(inputFilePath.toString())) {
								try {
									nc.deleteFile(inputFilePath.toString());
									expunged++;
								} catch (ServerFault e) {
									ctx.error("[{}] Unable to delete file, continue, but with ending error: {}", logId,
											e.getMessage());
									errorsExpunge.add(rec.uid);
								}
							}
						}
					}
					ctx.info("[{}] deleted expunged messages {}, {} errors: {}", logIdF, expunged, errorsExpunge.size(),
							errorsExpunge.stream().map(i -> String.valueOf(i)).collect(Collectors.joining(",")));
				} catch (ServerFault | IOException e) {
					ctx.info("[{}] Unable to load file '{}': {}", logId, cyrusIndexPath.toAbsolutePath().toString(),
							e.getMessage());
				}

				errorCount += errors.size();
			}
			Services services = initializeServices(new ResolvedMailboxInfo(domainUid, mbox));
			List<String> errorsRemaining = new ArrayList<>();
			if (nc.exists(inputLiveFolderPath.toString())) {
				processFolder(nc, inputLiveFolderPath, services, mailboxItems, folder, remaining, errorsRemaining);
			}
			if (nc.exists(inputArchiveFolderPath.toString())) {
				processFolder(nc, inputArchiveFolderPath, services, mailboxItems, folder, remaining, errorsRemaining);
			}
			ctx.info("[{}] migrated remaining messages {}, {} errors: {}", logIdF, remaining.longValue(),
					errorsRemaining.size(), errorsRemaining.stream().collect(Collectors.joining(",")));
			errorCount += errors.size();
		}
		return errorCount;

	}

	private void processFolder(INodeClient nc, Path inputLiveFolderPath, Services services,
			IDbMailboxRecords recordsApi, ItemValue<MailboxFolder> folder, AtomicLong remaining,
			List<String> errorsRemaining) {
		final int size = 500;
		Set<FileDescription> current;
		int errors;
		do {
			errors = 0;
			current = nc.pollFiles(inputLiveFolderPath.toString(), "^\\d+\\.$", size).stream()
					.filter(fd -> !fd.isDirectory() && fd.getName().endsWith(".")).collect(Collectors.toSet());
			for (FileDescription remainingFile : current) {
				try {
					ctx.info("Migrating remaining file {}", remainingFile.getPath());
					byte[] mailData = nc.read(remainingFile.getPath());
					String bodyGuid = Hashing.sha1().hashBytes(mailData).toString();
					Mail mail = new Mail(remainingFile.getPath(), mailData, new Date(remainingFile.getModified()),
							bodyGuid);
					importMail(services, recordsApi, folder, mail);
					nc.deleteFile(remainingFile.getPath());
					remaining.incrementAndGet();
				} catch (Exception e1) {
					ctx.error("Cannot migrate remaining file {}", remainingFile.getPath(), e1);
					errors++;
					errorsRemaining.add(remainingFile.getPath());
				}
			}
		} while (current.size() == size && errors != size);
	}

	private void importMail(Services services, IDbMailboxRecords recordsApi, ItemValue<MailboxFolder> resolvedFolder,
			Mail mail) throws IOException {
		ItemVersion added = createMail(services, recordsApi, resolvedFolder, mail);
		if (added.id <= 0) {
			ctx.error("Unable to inject message {}", mail.file);
		}
	}

	private ItemVersion createMail(Services services, IDbMailboxRecords recordsApi,
			ItemValue<MailboxFolder> resolvedFolder, Mail mail) throws IOException {
		AppendTx appendTx = services.mailboxApi.prepareAppend(resolvedFolder.internalId, 1);
		Date bodyDeliveryDate = mail.date == null ? new Date(appendTx.internalStamp) : mail.date;
		net.bluemind.core.api.Stream stream = GenericStream.simpleValue(mail.data, b -> b);
		services.bodiesApi.createWithDeliveryDate(mail.guid, bodyDeliveryDate.getTime(), stream);
		MessageBody messageBody = services.bodiesApi.getComplete(mail.guid);
		if (messageBody == null) {
			return new ItemVersion();
		}
		Set<String> references = (messageBody.references != null) ? Sets.newHashSet(messageBody.references)
				: Sets.newHashSet();
		Long conversationId = null;
		if (services.conversationReferenceApi != null) {
			conversationId = services.conversationReferenceApi.lookup(messageBody.messageId, references);
		}
		MailboxRecord rec = new MailboxRecord();
		rec.imapUid = appendTx.imapUid;
		rec.internalDate = bodyDeliveryDate;
		rec.messageBody = mail.guid;
		rec.conversationId = conversationId;
		rec.flags = List.of(MailboxItemFlag.System.Seen.value());
		rec.lastUpdated = rec.internalDate;
		return recordsApi.create(appendTx.imapUid + ".", rec);
	}

	private Services initializeServices(ResolvedMailboxInfo mailboxByEmail) {
		IMailboxFolders folderService = ctx.adminApi().instance(IMailboxFoldersByContainer.class, IMailReplicaUids
				.subtreeUid(mailboxByEmail.domainUid, mailboxByEmail.mailbox.value.type, mailboxByEmail.mailbox.uid));
		CyrusPartition cyrusPartition = CyrusPartition.forServerAndDomain(mailboxByEmail.mailbox.value.dataLocation,
				mailboxByEmail.domainUid);
		IDbReplicatedMailboxes mailboxApi = ctx.adminApi().instance(IDbByContainerReplicatedMailboxes.class,
				IMailReplicaUids.subtreeUid(mailboxByEmail.domainUid, mailboxByEmail.mailbox));
		IDbMessageBodies bodiesApi = ctx.adminApi().instance(IDbMessageBodies.class, cyrusPartition.name);
		IConversationReference conversationReferenceApi = null;
		if (mailboxByEmail.mailbox.value.type == Type.user) {
			conversationReferenceApi = ctx.adminApi().instance(IConversationReference.class, mailboxByEmail.domainUid,
					mailboxByEmail.mailbox.uid);
		}
		return new Services(mailboxApi, bodiesApi, conversationReferenceApi, folderService);

	}

	private static record ResolvedMailboxInfo(String domainUid, ItemValue<Mailbox> mailbox) {

	}

	private static record Services(IDbReplicatedMailboxes mailboxApi, IDbMessageBodies bodiesApi,
			IConversationReference conversationReferenceApi, IMailboxFolders foldersService) {
	}

	private static record Mail(String file, byte[] data, Date date, String guid) {

	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("mail");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return ResyncCyrusSpoolCommand.class;
		}
	}

}
