import { extensions } from "@bluemind/extensions";
import { pki } from "node-forge";

type Implementations = {
    getMyStatus: () => number;
    getMyEncryptionCertificate: (validity?: Date) => pki.Certificate | string;
    getMySignatureKeyPair: (validity?: Date) => {
        key: pki.rsa.PrivateKey;
        certificate: pki.Certificate;
    };
    getMyDecryptionKeyPair: (validity?: Date) => {
        key: pki.rsa.PrivateKey;
        certificate: pki.Certificate;
    };
    getAllDecryptionKeyPairs: (validity: Date) => {
        key: pki.rsa.PrivateKey;
        certificate: pki.Certificate;
    }[];
    decryptMimeEntity: (data: Blob, date: Date) => string;
    signMimeEntity: (entity: string) => string;
};
type ImplementationName = keyof Implementations;
type DecoratedImplementation<T extends ImplementationName> = (
    ...args: [...Parameters<Implementations[T]>, Implementations[T] | undefined]
) => ReturnType<Implementations[T]>;

type ImplementationExtension<T extends ImplementationName> = {
    fn: DecoratedImplementation<T>;
};

const cache = new Map<ImplementationName, Implementations[ImplementationName]>();
export default function <T extends ImplementationName>(name: T): Implementations[T] {
    if (!cache.has(name)) {
        const implementations = extensions.get("smime.implementations", name) as ImplementationExtension<T>[];

        const implementation = implementations.reverse().reduce(
            (previous, { fn }) =>
                function (...args: any[]) {
                    return fn.apply(null, [...(args as Parameters<Implementations[T]>), previous]);
                } as Implementations[T],
            undefined as Implementations[T] | undefined
        );
        cache.set(name, implementation as Implementations[T]);
    }
    return cache.get(name) as Implementations[T];
}
