/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */

package net.bluemind.core.auditlogs.client.loader.config;

import java.io.File;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

import net.bluemind.core.api.BMApi;
import net.bluemind.lifecycle.helper.SoftReset;

@BMApi(version = "3")
public class AuditLogConfig {
	private static final Logger logger = LoggerFactory.getLogger(AuditLogConfig.class);
	private static Holder instance = new Holder();
	public static final String OVERRIDE_PATH = "/etc/bm/auditlog-store.conf";

	protected AuditLogConfig() {

	}

	private static class Holder {
		private Config cur;

		public Holder() {
			cur = loadConfig();
			SoftReset.register(() -> cur = loadConfig());
		}

		public Config get() {
			return cur;
		}

	}

	public static class AuditLogStore {
		private AuditLogStore() {

		}

		public static final String ACTIVATED = "auditlog.activate";
		public static final String MULTIDOMAIN_DATASTREAMS = "auditlog.domain_datastream";
	}

	private static Config loadConfig() {
		Config conf = ConfigFactory.load(AuditLogConfig.class.getClassLoader(), "resources/auditlog-store.conf");
		String configPath = System.getProperty("auditlog.config.path");
		File local = configPath != null ? new File(configPath) : new File(OVERRIDE_PATH); // NOSONAR
		try {
			if (local.exists()) {
				Config parsed = ConfigFactory.parseFile(local);
				conf = parsed.withFallback(conf);
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}

		Config systemPropertyConfig = ConfigFactory.defaultApplication();
		return systemPropertyConfig.withFallback(conf);
	}

	public static Config get() {
		return instance.get();
	}

	protected static boolean getOrDefaultBool(String key) {
		try {
			return AuditLogConfig.get().getBoolean(key);
		} catch (Exception e) {
			return false;
		}
	}

	protected static String getOrDefaultStr(String key) {
		try {
			return AuditLogConfig.get().getString(key);
		} catch (Exception e) {
			return null;
		}
	}

	protected static int getOrDefaultInt(String key) {
		try {
			return AuditLogConfig.get().getInt(key);
		} catch (Exception e) {
			return 0;
		}
	}

	public static boolean isActivated() {
		return AuditLogConfig.getOrDefaultBool(AuditLogStore.ACTIVATED);
	}

	public static String getDataStreamName() {
		String dataStreamPattern = AuditLogConfig.getOrDefaultStr(AuditLogStore.MULTIDOMAIN_DATASTREAMS);
		return dataStreamPattern;
	}

	public static String resolveDataStreamName(String domainUid) {

		String dataStreamPattern = AuditLogConfig.getOrDefaultStr(AuditLogStore.MULTIDOMAIN_DATASTREAMS);

		if (dataStreamPattern.contains("%s")) {
			return String.format(dataStreamPattern, domainUid);
		}
		return dataStreamPattern;
	}

	@VisibleForTesting
	public static void clear() {
		instance = new Holder();
	}
}
