/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.webappdata.cql.store;

import java.sql.SQLException;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.core.container.cql.store.CqlItemStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.repository.sequences.SequenceStores;
import net.bluemind.webappdata.api.WebAppData;
import net.bluemind.webappdata.repository.IWebAppDataStore;

public class CqlWebAppDataStore extends CqlAbstractStore implements IWebAppDataStore {

	private long contId;
	private final CqlItemStore itemStore;

	public CqlWebAppDataStore(CqlSession s, Container cont) {
		super(s);
		this.itemStore = new CqlItemStore(cont, s, SequenceStores.getDefault(), null);
		this.contId = cont.id;
	}

	@Override
	public void create(Item item, WebAppData value) throws SQLException {
		update(item, value);
	}

	@Override
	public void update(Item item, WebAppData value) throws SQLException {
		voidCql("""
				update t_webappdata
				set value=?
				where container_id=? and item_id=?
				""", value.value, contId, item.id);
	}

	@Override
	public void delete(Item item) throws SQLException {
		voidCql("delete from t_webappdata where container_id=? and item_id=?", contId, item.id);
	}

	@Override
	public WebAppData get(Item item) throws SQLException {
		return unique("select value from t_webappdata where container_id=? and item_id=?",
				r -> WebAppData.create(item.uid, r.getString(0)), voidPop(), contId, item.id);
	}

	@Override
	public void deleteAll() throws SQLException {
		// TODO Auto-generated method stub

	}

	@Override
	public boolean exists(Item item) throws SQLException {
		return get(item) != null;
	}

	@Override
	public WebAppData getByKey(String key) throws SQLException {
		Item item = itemStore.get(key);
		if (item == null) {
			return null;
		}
		return get(item);
	}

}
