/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.system.ldap.importation.internal.tools;

import java.util.Locale;
import java.util.Map;

import org.apache.directory.api.ldap.model.exception.LdapInvalidDnException;
import org.apache.directory.api.ldap.model.name.Dn;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.lib.ldap.LdapProtocol;
import net.bluemind.system.importation.commons.tools.ParametersValidator;
import net.bluemind.system.ldap.importation.api.LdapProperties;
import net.bluemind.system.ldap.importation.internal.l10n.Messages;

public class LdapParametersValidator extends ParametersValidator {
	/**
	 * @param current
	 * @param locale
	 * @throws ServerFault
	 */
	public void validate(Map<String, String> current, Locale locale) {
		if (Boolean.FALSE.equals(Boolean.valueOf(current.get(LdapProperties.import_ldap_enabled.name())))) {
			return;
		}

		checkLdapHostname(current.get(LdapProperties.import_ldap_hostname.name()), locale);
		checkLdapProtocol(current.get(LdapProperties.import_ldap_protocol.name()), locale);
		checkLdapAllCertificate(current.get(LdapProperties.import_ldap_accept_certificate.name()));
		checkLdapLoginDn(current.get(LdapProperties.import_ldap_login_dn.name()), locale);
		checkLdapBaseDn(current.get(LdapProperties.import_ldap_base_dn.name()), locale);
		checkLdapUserFilter(current.get(LdapProperties.import_ldap_user_filter.name()), locale);
		checkLdapGroupFilter(current.get(LdapProperties.import_ldap_group_filter.name()), locale);
	}

	/**
	 * @param allCertificate
	 * @throws ServerFault
	 */
	public void checkLdapAllCertificate(String allCertificate) {
		if (allCertificate == null) {
			return;
		}

		if (allCertificate.equalsIgnoreCase("true") || allCertificate.equalsIgnoreCase("false")) {
			return;
		}

		throw new ServerFault("All certificate value must be null, true or false", ErrorCode.INVALID_PARAMETER);
	}

	/**
	 * @param protocol
	 * @param locale
	 * @throws ServerFault
	 */
	public void checkLdapProtocol(String protocol, Locale locale) {
		if (protocol == null) {
			throw new ServerFault(Messages.get(locale).nullLdapProtocol(), ErrorCode.INVALID_PARAMETER);
		}

		try {
			LdapProtocol.getProtocol(protocol);
		} catch (IllegalArgumentException i) {
			throw new ServerFault(Messages.get(locale).invalidProtocol() + ": " + protocol,
					ErrorCode.INVALID_PARAMETER);
		}
	}

	/**
	 * @param ldapHostname
	 * @param locale
	 * @throws ServerFault
	 */
	public void checkLdapHostname(String ldapHostname, Locale locale) {
		if (ldapHostname == null) {
			throw new ServerFault(Messages.get(locale).invalidHostname(), ErrorCode.INVALID_PARAMETER);
		}

		if (ldapHostname.trim().isEmpty()) {
			throw new ServerFault(Messages.get(locale).invalidHostname(), ErrorCode.INVALID_PARAMETER);
		}

		long sepCount = ldapHostname.chars().filter(ch -> ch == ':').count();
		if (sepCount > 1) {
			throw new ServerFault(Messages.get(locale).invalidHostname(), ErrorCode.INVALID_PARAMETER);
		}

		if (sepCount == 1) {
			try {
				Integer port = Integer.valueOf(ldapHostname.split(":")[1]);
				if (port <= 0 || port > 65535) {
					throw new ServerFault(Messages.get(locale).invalidPort(), ErrorCode.INVALID_PARAMETER);
				}
			} catch (NumberFormatException nfe) {
				throw new ServerFault(Messages.get(locale).invalidPort(), ErrorCode.INVALID_PARAMETER);
			}
		}
	}

	/**
	 * @param baseDn
	 * @param locale
	 * @throws ServerFault
	 */
	public void checkLdapBaseDn(String baseDn, Locale locale) {
		if (baseDn == null) {
			return;
		}

		try {
			new Dn(baseDn);
		} catch (LdapInvalidDnException e) {
			throw new ServerFault(Messages.get(locale).invalidBaseDn(), ErrorCode.INVALID_PARAMETER);
		}
	}

	/**
	 * @param loginDn
	 * @param locale
	 * @throws ServerFault
	 */
	public void checkLdapLoginDn(String loginDn, Locale locale) {
		if (loginDn == null) {
			return;
		}

		try {
			new Dn(loginDn);
		} catch (LdapInvalidDnException e) {
			throw new ServerFault(Messages.get(locale).invalidLoginDn(), ErrorCode.INVALID_PARAMETER);
		}
	}

	/**
	 * @param userFilter
	 * @param locale
	 * @throws ServerFault
	 */
	public void checkLdapUserFilter(String userFilter, Locale locale) {
		checkFilter(userFilter, Messages.get(locale).invalidUserFilter());
	}

	public void checkLdapGroupFilter(String groupFilter, Locale locale) {
		checkFilter(groupFilter, Messages.get(locale).invalidGroupFilter());
	}

	/**
	 * @param previous
	 * @param current
	 * @throws ServerFault
	 */
	public void noChanges(Map<String, String> previous, Map<String, String> current) {
		unchanged("Domain admin can't disable LDAP", LdapProperties.import_ldap_enabled.name(), previous, current);
		unchanged("Domain admin can't update LDAP server hostname", LdapProperties.import_ldap_hostname.name(),
				previous, current);
		unchanged("Domain admin can't update LDAP protocol", LdapProperties.import_ldap_protocol.name(), previous,
				current);
		unchanged("Domain admin can't update LDAP all certificate",
				LdapProperties.import_ldap_accept_certificate.name(), previous, current);
		unchanged("Domain admin can't update LDAP base DN", LdapProperties.import_ldap_base_dn.name(), previous,
				current);
		unchanged("Domain admin can't update LDAP user login", LdapProperties.import_ldap_login_dn.name(), previous,
				current);
		unchanged("Domain admin can't update LDAP user password", LdapProperties.import_ldap_password.name(), previous,
				current);
		unchanged("Domain admin can't update LDAP external ID", LdapProperties.import_ldap_ext_id_attribute.name(),
				previous, current);
		unchanged("Domain admin can't update LDAP split domain group",
				LdapProperties.import_ldap_relay_mailbox_group.name(), previous, current);
		unchanged("Domain admin can't update LDAP users filter", LdapProperties.import_ldap_user_filter.name(),
				previous, current);
		unchanged("Domain admin can't update LDAP groups filter", LdapProperties.import_ldap_group_filter.name(),
				previous, current);
		unchanged("Domain admin can't update LDAP last update", LdapProperties.import_ldap_lastupdate.name(), previous,
				current);
	}

	/**
	 * @param errorMessage
	 * @param propertyName
	 * @param previousProperties
	 * @param properties
	 * @throws ServerFault
	 */
	private void unchanged(String errorMessage, String propertyName, Map<String, String> previousProperties,
			Map<String, String> properties) {
		String previousValue = previousProperties.get(propertyName);
		String value = properties.get(propertyName);

		if (previousValue == null && value == null) {
			return;
		}

		if (previousValue != null && previousValue.equals(value)) {
			return;
		}

		throw new ServerFault(errorMessage, ErrorCode.INVALID_PARAMETER);
	}
}
