/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.system.importation.commons.pool;

import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.message.BindRequest;
import org.apache.directory.api.ldap.model.message.BindRequestImpl;
import org.apache.directory.api.ldap.model.message.BindResponse;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;
import org.apache.directory.ldap.client.api.LdapConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.system.importation.commons.Parameters;

public class LdapPoolByDomain {
	private static final Logger logger = LoggerFactory.getLogger(LdapPoolByDomain.class);
	private static final ConcurrentHashMap<Parameters, LdapPoolWrapper> poolByDomain = new ConcurrentHashMap<>();

	/**
	 * Get anonymous LDAP connection
	 * 
	 * @param parameters
	 * @return
	 * @throws LdapException
	 */
	public Optional<LdapConnectionContext> getConnectionContext(Parameters parameters) {
		if (poolByDomain.putIfAbsent(parameters, new LdapPoolWrapper(parameters)) == null && logger.isDebugEnabled()) {
			logger.debug("Initialize LDAP pool for: {}", parameters);
		}

		return poolByDomain.get(parameters).getConnection();
	}

	/**
	 * Get authenticated LDAP connection, using login/password from LDAP parameters
	 * 
	 * @param parameters
	 * @return
	 * @throws LdapException
	 * @throws Exception
	 */
	public Optional<LdapConnectionContext> getAuthenticatedConnectionContext(Parameters parameters) {
		if (parameters.ldapServer.login == null || parameters.ldapServer.login.isEmpty()) {
			return getConnectionContext(parameters);
		}

		return getConnectionContext(parameters).map(this::authenticate);
	}

	private LdapConnectionContext authenticate(LdapConnectionContext ldapConnectionContext) {
		BindRequest bindRequest = new BindRequestImpl();
		bindRequest.setSimple(true);
		bindRequest.setName(ldapConnectionContext.parameters.ldapServer.login);
		bindRequest.setCredentials(ldapConnectionContext.parameters.ldapServer.password);

		long ldSearchTime = System.currentTimeMillis();
		BindResponse response = null;
		try {
			response = ldapConnectionContext.ldapCon.bind(bindRequest);
		} catch (LdapException le) {
			logger.error(le.getMessage(), le);
			releaseConnectionContext(ldapConnectionContext.setError());
			return null;
		}

		ldSearchTime = System.currentTimeMillis() - ldSearchTime;

		if (ResultCodeEnum.SUCCESS != response.getLdapResult().getResultCode()
				|| !ldapConnectionContext.ldapCon.isAuthenticated()) {
			String errorMsg = "Fail to bind on: " + ldapConnectionContext.parameters.toString() + " (" + ldSearchTime
					+ "ms)";
			if (response.getLdapResult().getDiagnosticMessage() != null
					&& !response.getLdapResult().getDiagnosticMessage().isEmpty()) {
				errorMsg += " - " + response.getLdapResult().getDiagnosticMessage();
			}

			releaseConnectionContext(ldapConnectionContext);

			logger.error(errorMsg);
			return null;
		}

		if (logger.isDebugEnabled()) {
			logger.debug("Bind success on: {} ({}ms)", ldapConnectionContext.parameters, ldSearchTime);
		}

		return ldapConnectionContext;
	}

	public void releaseConnectionContext(LdapConnectionContext ldapConCtx) {
		getDomainPoolWrapperOrCloseLdapConnection(ldapConCtx.parameters, ldapConCtx.ldapCon)
				.ifPresent(poolWrapper -> poolWrapper.doReleaseConnection(ldapConCtx));
	}

	private Optional<LdapPoolWrapper> getDomainPoolWrapperOrCloseLdapConnection(Parameters ldapParameters,
			LdapConnection ldapCon) {
		LdapPoolWrapper pool = poolByDomain.get(ldapParameters);

		if (pool != null) {
			return Optional.of(pool);
		}

		logger.warn("No LDAP connection pool for: {}, closing connection", ldapParameters);
		try {
			ldapCon.close();
		} catch (IOException ioe) {
			// No pool found, and unable to close orphan LDAP connection...
			logger.warn("Unable to close LDAP connection for {}", ldapParameters, ioe);
		}

		return Optional.empty();
	}
}
