/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.notes.incremental;

import java.util.List;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.DataSourceRouter;
import net.bluemind.core.rest.BmContext;
import net.bluemind.indexing.incremental.AbstractTypeIndexer;
import net.bluemind.indexing.incremental.AbstractTypeIndexer.IndexWriter;
import net.bluemind.indexing.incremental.TypeIndexerFactory;
import net.bluemind.lib.elasticsearch.ESearchActivator;
import net.bluemind.notes.api.INote;
import net.bluemind.notes.api.INoteUids;
import net.bluemind.notes.api.VNote;
import net.bluemind.notes.persistence.VNoteIndexStore;

public class VNoteIncrementalIndexFactory implements TypeIndexerFactory {

	@Override
	public String type() {
		return INoteUids.TYPE;
	}

	@Override
	public TypeIndexer create(BmContext ctx, Container c) {
		INote notesApi = ctx.provider().instance(INote.class, c.uid);
		ElasticsearchClient esClient = ESearchActivator.getClient();
		VNoteIndexStore indexStore = new VNoteIndexStore(esClient, c, DataSourceRouter.location(ctx, c.uid));
		return new NoteIndexer(notesApi, new INoteIndexWriter(indexStore));
	}

	public static class INoteIndexWriter implements IndexWriter<VNote> {

		private VNoteIndexStore indexStore;

		public INoteIndexWriter(VNoteIndexStore cidx) {
			this.indexStore = cidx;
		}

		@Override
		public void write(ItemValue<VNote> toIndex) {
			indexStore.update(toIndex.item(), toIndex.value);

		}

		@Override
		public void deleteByIds(List<Long> itemIds) {
			itemIds.forEach(item -> indexStore.delete(item));

		}

	}

	public static class NoteIndexer extends AbstractTypeIndexer<VNote> {

		public NoteIndexer(INote note, IndexWriter<VNote> writer) {
			super(note, () -> note, writer);
		}

	}

}
