/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.milter.metrics;

import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.field.address.LenientAddressBuilder;
import org.apache.james.mime4j.stream.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Registry;

import io.netty.buffer.Unpooled;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.domain.api.Domain;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.milter.IMilterListener;
import net.bluemind.milter.MilterHeaders;
import net.bluemind.milter.MilterInstanceID;
import net.bluemind.milter.SmtpEnvelope;
import net.bluemind.milter.Status;
import net.bluemind.milter.cache.DomainAliasCache;

public class InboundOutboundClassifier implements IMilterListener {

	public enum TrafficClass {
		INTERNAL, EXTERNAL;
	}

	public static class ClassifiedAddress {
		public final String email;
		public final TrafficClass traficClass;

		public ClassifiedAddress(String email, TrafficClass klass) {
			this.email = email;
			this.traficClass = klass;
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(InboundOutboundClassifier.class);

	private ClassifiedAddress from;
	private final List<ClassifiedAddress> recipients;
	private final Registry registry;
	private final IdFactory idFactory;
	private int size;

	private Optional<ClassifiedAddress> classify(String email) {
		Mailbox parsed = LenientAddressBuilder.DEFAULT.parseMailbox(email);
		if (parsed == null) {
			logger.warn("mime4j cannot parse email '{}'", email);
			return Optional.empty();
		}
		ItemValue<Domain> domain = DomainAliasCache.getDomain(parsed.getDomain());
		if (domain != null) {
			return Optional.of(new ClassifiedAddress(parsed.getAddress(), TrafficClass.INTERNAL));
		} else {
			return Optional.of(new ClassifiedAddress(parsed.getAddress(), TrafficClass.EXTERNAL));
		}
	}

	public InboundOutboundClassifier(Registry registry, IdFactory idFactory) {
		this.recipients = new LinkedList<>();
		this.registry = registry;
		this.idFactory = idFactory;
	}

	@Override
	public Status onEnvFrom(Properties properties, String envFrom) {
		from = classify(envFrom).orElse(null);
		return Status.getContinue();
	}

	@Override
	public Status onEnvRcpt(Properties properties, String rcpt) {
		classify(rcpt).ifPresent(recipients::add);
		return Status.getContinue();
	}

	@Override
	public Status onHeader(String headerf, String headerv) {
		return Status.getContinue();
	}

	@Override
	public Status onEoh() {
		return Status.getContinue();
	}

	@Override
	public Status onBody(ByteBuffer bodyp) {
		this.size = Unpooled.wrappedBuffer(bodyp).readableBytes();
		return Status.getContinue();
	}

	@Override
	public Status onMessage(Properties properties, SmtpEnvelope envelope, Message message) {
		Field handled = message.getHeader().getField(MilterHeaders.HANDLED);
		if ((handled == null || handled.getBody().equals(MilterInstanceID.get())) && from != null) {
			long inbound = 0;
			long outbound = 0;
			long internal = 0;
			if (from.traficClass == TrafficClass.INTERNAL) {
				outbound = recipients.stream().filter(r -> r.traficClass == TrafficClass.EXTERNAL).count();
				internal = recipients.stream().filter(r -> r.traficClass == TrafficClass.INTERNAL).count();
			} else if (from.traficClass == TrafficClass.EXTERNAL) {
				inbound = recipients.stream().filter(r -> r.traficClass == TrafficClass.INTERNAL).count();
			}
			Counter inboundCounter = registry.counter(idFactory.name("class", "type", "INBOUND"));
			Counter inboundSizeCounter = registry.counter(idFactory.name("size", "type", "INBOUND"));
			Counter internalCounter = registry.counter(idFactory.name("class", "type", "INTERNAL"));
			Counter internalSizeCounter = registry.counter(idFactory.name("size", "type", "INTERNAL"));
			Counter outboundCounter = registry.counter(idFactory.name("class", "type", "OUTBOUND"));
			Counter outboundSizeCounter = registry.counter(idFactory.name("size", "type", "OUTBOUND"));

			// global traffic count
			registry.counter(idFactory.name("processed-msg")).increment();
			registry.counter(idFactory.name("processed-size")).increment(size);

			if (inbound > 0) {
				inboundCounter.increment(inbound);
				inboundSizeCounter.increment(inbound * size);
			}
			if (outbound > 0) {
				outboundCounter.increment(outbound);
				outboundSizeCounter.increment(outbound * size);
			}
			if (internal > 0) {
				internalCounter.increment(internal);
				internalSizeCounter.increment(internal * size);
			}
		}
		return Status.getContinue();
	}

}
