/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.milter.action.delegation;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.columba.ristretto.message.Address;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.api.Regex;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.mailflow.rbe.IClientContext;
import net.bluemind.milter.Status;
import net.bluemind.milter.SysconfHelper;
import net.bluemind.milter.action.MilterAction;
import net.bluemind.milter.action.MilterActionException;
import net.bluemind.milter.action.MilterActionsFactory;
import net.bluemind.milter.action.UpdatedMailMessage;
import net.bluemind.milter.cache.DirectoryCache;
import net.bluemind.milter.cache.DomainAliasCache;
import net.bluemind.role.api.BasicRoles;
import net.bluemind.user.api.IUser;

public class DelegationAction implements MilterAction {
	private static final Logger logger = LoggerFactory.getLogger(DelegationAction.class);
	private static final Status SMTP_ERROR_STATUS = Status.getCustom("550", "5.7.1",
			new String[] { "Message cannot be delivered because of insufficient delegation rights." });

	public static class DelegationActionFactory implements MilterActionsFactory {

		@Override
		public MilterAction create() {
			return new DelegationAction();
		}
	}

	@Override
	public String identifier() {
		return "milter.delegation";
	}

	@Override
	public String description() {
		return "Milter delegation action";
	}

	@Override
	public void execute(UpdatedMailMessage modifier, Map<String, String> configuration,
			Map<String, String> evaluationData, IClientContext context) {
		ItemValue<Domain> contextDomain = context.getSenderDomain();
		String contextAliasDomain = contextDomain.value.defaultAlias;
		ItemValue<Domain> domainItem = DomainAliasCache.getDomain(contextAliasDomain);
		if (domainItem == null) {
			String msg = String.format("Cannot find domain/alias %s", contextDomain);
			logger.warn(msg);
			throw new MilterActionException(msg);
		}

		if (!modifier.getMessage().getFrom().isEmpty()) {
			resolveConnectedUserAddress(modifier, context)
					.ifPresent(info -> verifyAclAndApplyHeader(modifier, context, info));
		}

	}

	private boolean isAdmin(String senderAddress) {
		return "admin0@global.virt".equals(senderAddress);
	}

	private void verifyAclAndApplyHeader(UpdatedMailMessage modifier, IClientContext context,
			DelegationHeaderInfo info) {
		try {
			DirectoryCache.getUserUidByEmail(context, info.senderDomain.uid, info.sender).ifPresentOrElse(
					senderUserUid -> DirectoryCache.getUserUidByEmail(context, info.senderDomain.uid, info.from)
							.ifPresentOrElse(fromUserUid -> {
								if (!senderUserUid.equals(fromUserUid)) {
									canSendAsOnBehalf(context, modifier, fromUserUid, info, senderUserUid);
								}
							}, () -> verifySenderCanUseExternalIdentity(senderUserUid, modifier, context, info.sender)),
					() -> logger.error("User (email sender) matching to address '{}' not found", info.sender));
		} catch (ServerFault e) {
			if (e.getCode() == ErrorCode.INVALID_PARAMETER) {
				modifier.errorStatus = SMTP_ERROR_STATUS;
				String errorMsg = """
						Message cannot be delivered because one of these login email have not been found
						Sender: '%s'
						From: '%s'

						Return SMTP Status: %s
						""".formatted(info.sender, info.from, SMTP_ERROR_STATUS);
				logger.error(errorMsg);
			} else {
				modifier.errorStatus = SMTP_ERROR_STATUS;
				String errorMsg = """
						Message cannot be delivered
						Sender: '%s'
						From: '%s'

						Return SMTP Status: %s
						Error message: %s
						""".formatted(info.sender, info.from, SMTP_ERROR_STATUS, e.getMessage());
				logger.error(errorMsg);
				StringWriter writer = new StringWriter();
				PrintWriter printWriter = new PrintWriter(writer);
				e.printStackTrace(printWriter);
				logger.error(writer.toString());
			}
		}
	}

	private void verifySenderCanUseExternalIdentity(String senderUserUid, UpdatedMailMessage modifier,
			IClientContext context, String senderAddress) {
		if (!hasRole(context, senderUserUid)) {
			modifier.errorStatus = SMTP_ERROR_STATUS;
			String msg = """
					Message cannot be delivered because '%s' has no been found as an External Identity

					Return SMTP Status: %s
					""".formatted(senderAddress, SMTP_ERROR_STATUS);
			logger.error(msg);
		}
	}

	private boolean hasRole(IClientContext context, String senderUserUid) {
		return context.provider().instance(IUser.class, context.getSenderDomain().uid) //
				.getResolvedRoles(senderUserUid).stream()
				.anyMatch(role -> role.equals(BasicRoles.ROLE_EXTERNAL_IDENTITY));
	}

	private Optional<DelegationHeaderInfo> resolveConnectedUserAddress(UpdatedMailMessage modifier,
			IClientContext context) {

		Optional<String> authentLogin = Optional.ofNullable(modifier.properties.get("{auth_authen}"))
				.map(authAuthens -> authAuthens.stream().map(Strings::emptyToNull).filter(Objects::nonNull).findFirst()
						.orElse(null));

		Optional<String> sender = authentLogin.filter(a -> !Regex.EMAIL.validate(a))
				.map(a -> DomainAliasCache.getDomainFromEmail(a).map(DomainAliasCache::getDomainAlias)
						.orElseGet(() -> DomainAliasCache.getDomainAlias(SysconfHelper.defaultDomain.get())))
				.map(d -> Optional.ofNullable(authentLogin.get().concat("@").concat(d))).orElse(authentLogin);

		return sender.map(senderAddress -> {
			if (isAdmin(senderAddress)) {
				return null;
			}

			String fromAddress = modifier.getMessage().getFrom().get(0).getAddress();
			if (!senderAddress.equalsIgnoreCase(fromAddress)) {
				return new DelegationHeaderInfo(senderAddress, fromAddress, context.getSenderDomain());
			}

			return null;
		});
	}

	private void canSendAsOnBehalf(IClientContext context, UpdatedMailMessage modifier, String fromUid,
			DelegationHeaderInfo info, String senderUserUid) {
		String msg = """
				Try to send a message using delegation:
				Domain: %s
				Sender Address: %s
				From Address: %s
				""".formatted(info.senderDomain, info.sender, info.from);
		logger.info(msg);

		if (context.provider().instance(IUser.class, context.getSenderDomain().uid).getResolvedRoles(senderUserUid)
				.contains(BasicRoles.ROLE_GLOBAL_SEND_AS)) {
			addSenderHeader(context, modifier, info);
			return;
		}

		List<Verb> filteredVerbs = context.sudo(info.sender, context.getSenderDomain().uid)
				.instance(IContainerManagement.class, IMailboxAclUids.uidForMailbox(fromUid))
				.canAccessVerbs(List.of(Verb.SendOnBehalf.name(), Verb.SendAs.name())).getVerbs().stream()
				.map(Verb::valueOf).toList();

		if (filteredVerbs.isEmpty()) {
			insufficientDelegationRights(modifier, info);
		} else {
			if (filteredVerbs.stream().noneMatch(v -> v.can(Verb.SendAs))) {
				addSenderHeader(context, modifier, info);
			}
		}
	}

	private void insufficientDelegationRights(UpdatedMailMessage modifier, DelegationHeaderInfo info) {
		modifier.errorStatus = SMTP_ERROR_STATUS;
		String errorMsg = """
				Message cannot be delivered because of insufficient delegation rights
				Sender: '%s'
				From: '%s'

				Return SMTP Status: %s
				""".formatted(info.sender, info.from, SMTP_ERROR_STATUS);
		logger.error(errorMsg);
	}

	private void addSenderHeader(IClientContext context, UpdatedMailMessage modifier, DelegationHeaderInfo info) {
		try {
			DomainAliasCache.getLeftPartFromEmail(info.sender).ifPresentOrElse(localPart -> {
				Optional<VCard> vCard = DirectoryCache.getVCard(context, info.senderDomain.uid, info.sender);
				String displayname = vCard.isPresent() && vCard.get().identification != null
						&& vCard.get().identification.formatedName != null
								? vCard.get().identification.formatedName.value
								: localPart;
				String domainPart = DomainAliasCache.getDomainFromEmail(info.sender)
						.filter(d -> !d.equals(info.senderDomain.uid) && info.senderDomain.value.aliases.contains(d))
						.orElse(info.senderDomain.value.defaultAlias);
				Address senderAddress = new Address(displayname, localPart + "@" + domainPart);
				modifier.addHeader("Sender", senderAddress.toString(), identifier());
			}, () -> modifier.addHeader("Sender", info.sender, identifier()));

		} catch (Exception e) {
			throw new MilterActionException(e);
		}
	}

	private record DelegationHeaderInfo(String sender, String from, ItemValue<Domain> senderDomain) {
	}

}
