/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.memory.pool.cli;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.LongAdder;

import com.typesafe.config.ConfigFactory;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.memory.pool.api.IChunk;
import net.bluemind.memory.pool.api.IMMapPoolConfig;
import net.bluemind.memory.pool.api.MemoryMappedPool;
import net.bluemind.memory.pool.api.MemoryMappedPool.PoolStats;
import net.bluemind.memory.pool.api.Pools;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

@Command(name = "mmap-pool-sim", mixinStandardHelpOptions = true, version = "mmap-pool simulation 1.0", description = "Tests mmap pool")
public class Simulation implements ICmdLet, Runnable {

	@Parameters(description = "The directory where mmap-pool will store its files", defaultValue = "mmap-pool")
	private File directory; // picocli convertit automatiquement le String en File

	@Option(names = { "-c", "--concurrency" }, description = "Concurrent users", defaultValue = "100")
	private int concurrency; // picocli convertit automatiquement en int

	@Option(names = { "-l", "--loops" }, description = "per user loops", defaultValue = "50")
	private int cycles;

	@Option(names = { "-s", "--segment" }, description = "segment size", defaultValue = "256M")
	private String segmentSize;

	@Option(names = { "--chunk" }, description = "max-chunk-size", defaultValue = "32M")
	private String maxChunkSize;

	@Option(names = { "--locked" }, description = "max-locked-memory", defaultValue = "2G")
	private String maxLockedMemory;

	private CliContext ctx;

	@Override
	public void run() {
		IMMapPoolConfig config = configFromParams();

		try (MemoryMappedPool pool = Pools.factory().create(config)) {
			simulateOn(pool);
			PoolStats stats = pool.getStats();
			ctx.info("{}", stats);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		} catch (Exception e) {
			throw new CliException(e);
		}
	}

	private IMMapPoolConfig configFromParams() {
		return new IMMapPoolConfig() {

			@Override
			public long maxLockedInMemory() {
				return readSize(maxLockedMemory);
			}

			@Override
			public boolean isEnableStats() {
				return true;
			}

			@Override
			public long getSegmentSize() {
				return readSize(segmentSize);
			}

			@Override
			public long getMaxChunkSize() {
				return readSize(maxChunkSize);
			}

			@Override
			public Path getBaseDirectory() {
				return directory.toPath();
			}

			private long readSize(String sizeWithUnit) {
				return ConfigFactory.parseString("size: " + sizeWithUnit).getBytes("size");
			}
		};
	}

	private void simulateOn(MemoryMappedPool pool) throws InterruptedException {
		List<Thread> pending = new ArrayList<>();
		byte[] rand128k = new byte[128 * K];
		byte[] rand1M = new byte[1 * M];
		byte[] rand4M = new byte[4 * M];
		ThreadLocalRandom.current().nextBytes(rand128k);
		ThreadLocalRandom.current().nextBytes(rand1M);
		ThreadLocalRandom.current().nextBytes(rand4M);
		LongAdder total = new LongAdder();
		for (int i = 0; i < concurrency; i++) {
			Thread spwaned = Thread.ofVirtual().name("sim-" + i).start(() -> {
				try {
					for (int j = 0; j < cycles; j++) {
						storeChunksThenReadBack(total, pool, rand128k, rand1M, rand4M);
					}
				} catch (Exception e) {
					e.printStackTrace();
					System.exit(1);
				}
			});
			pending.add(spwaned);
		}
		ctx.info("Joining...");
		for (Thread thread : pending) {
			thread.join();
		}
		ctx.info("Did {}GB of reads", total.sum() / G);
	}

	private void storeChunksThenReadBack(LongAdder totalRead, MemoryMappedPool pool, byte[] rand128k, byte[] rand1m,
			byte[] rand4m) throws IOException, InterruptedException {
		IChunk small = pool.allocateFrom(rand128k);
		Thread.sleep(1);
		IChunk medium = pool.allocateFrom(rand1m);
		Thread.sleep(1);
		IChunk large = pool.allocateFrom(rand4m);
		Thread.sleep(1);

		reReadChunk(totalRead, small);
		reReadChunk(totalRead, medium);
		reReadChunk(totalRead, large);
	}

	private void reReadChunk(LongAdder totalRead, IChunk small) throws IOException, InterruptedException {
		try (InputStream in = small.openStream(); OutputStream out = OutputStream.nullOutputStream()) {
			long transferred = in.transferTo(out);
			totalRead.add(transferred);
			Thread.sleep(1);
		} finally {
			small.release();
		}
	}

	private static final int K = 1024;
	private static final int M = 1024 * K;
	private static final int G = 1024 * M;

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("inject");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return Simulation.class;
		}

	}

}
