/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2026
 *
 * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License)
 * or the CeCILL as published by CeCILL.info (version 2 of the License).
 *
 * There are special exceptions to the terms and conditions of the
 * licenses as they are applied to this program. See LICENSE.txt in
 * the directory of this program distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.lib.elasticsearch.client.transport;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import co.elastic.clients.transport.DefaultTransportOptions;
import co.elastic.clients.transport.TransportOptions;
import co.elastic.clients.transport.http.HeaderMap;

public final class JdkHttpClientOptions extends DefaultTransportOptions {
	private final Duration requestTimeout;

	private JdkHttpClientOptions(Builder builder) {
		super(builder.headers, builder.parameters, builder.onWarnings);
		this.requestTimeout = builder.requestTimeout;
	}

	public Duration requestTimeout() {
		return requestTimeout;
	}

	public static Builder builder() {
		return new Builder();
	}

	public static JdkHttpClientOptions of(TransportOptions options) {
		if (options instanceof JdkHttpClientOptions jdkOptions) {
			return jdkOptions;
		}
		if (options == null) {
			return builder().build();
		}
		Builder builder = builder();
		for (Map.Entry<String, String> header : options.headers()) {
			builder.addHeader(header.getKey(), header.getValue());
		}
		for (Map.Entry<String, String> param : options.queryParameters().entrySet()) {
			builder.setParameter(param.getKey(), param.getValue());
		}
		if (options.onWarnings() != null) {
			builder.onWarnings(options.onWarnings());
		}
		return builder.build();
	}

	public static final class Builder implements TransportOptions.Builder {

		private HeaderMap headers;
		private java.util.Map<String, String> parameters;
		private Function<List<String>, Boolean> onWarnings;
		private Duration requestTimeout = Duration.ofSeconds(30);

		public Builder() {
		}

		public Builder requestTimeout(Duration timeout) {
			this.requestTimeout = timeout;
			return this;
		}

		@Override
		public Builder addHeader(String name, String value) {
			if (name.equalsIgnoreCase(HeaderMap.CLIENT_META)) {
				return this;
			}
			if (this.headers == null) {
				this.headers = new HeaderMap();
			}
			headers.add(name, value);
			return this;
		}

		@Override
		public Builder setHeader(String name, String value) {
			if (name.equalsIgnoreCase(HeaderMap.CLIENT_META)) {
				return this;
			}
			if (this.headers == null) {
				this.headers = new HeaderMap();
			}
			headers.put(name, value);
			return this;
		}

		@Override
		public Builder removeHeader(String name) {
			if (this.headers != null) {
				headers.remove(name);
			}
			return this;
		}

		@Override
		public Builder setParameter(String name, String value) {
			if (parameters == null) {
				parameters = new java.util.HashMap<>();
			}
			parameters.put(name, value);
			return this;
		}

		@Override
		public Builder removeParameter(String name) {
			if (parameters != null) {
				parameters.remove(name);
			}
			return this;
		}

		@Override
		public Builder onWarnings(Function<List<String>, Boolean> listener) {
			this.onWarnings = listener;
			return this;
		}

		@Override
		public JdkHttpClientOptions build() {
			return new JdkHttpClientOptions(this);
		}
	}
}
