/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2026
 *
 * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License)
 * or the CeCILL as published by CeCILL.info (version 2 of the License).
 *
 * There are special exceptions to the terms and conditions of the
 * licenses as they are applied to this program. See LICENSE.txt in
 * the directory of this program distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.lib.elasticsearch.client.transport;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.netflix.spectator.api.Clock;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.Timer;

import co.elastic.clients.transport.TransportOptions;
import co.elastic.clients.transport.http.TransportHttpClient;
import co.elastic.clients.util.BinaryData;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;

public final class JdkHttpClient implements TransportHttpClient {
	private static final Logger logger = LoggerFactory.getLogger(JdkHttpClient.class);

	private final HttpClient httpClient;
	private final NodePool nodePool;
	private final ExecutorService virtualThreadExecutor;
	private final Semaphore connectionLimiter;
	private final Duration requestTimeout;
	private final Duration connectionAcquireTimeout;
	private final Map<String, String> defaultHeaders;
	private static final Registry registry = MetricsRegistry.get();
	private static final IdFactory idFactory = new IdFactory("elasticsearch-http-client", registry,
			JdkHttpClient.class);
	private static final Timer connectionAcquireLatency = registry.timer(idFactory.name("connection-acquire-latency"));
	private static final Counter requestsSuccesses = registry
			.counter(idFactory.name("requests").withTag("success", true));
	private static final Counter requestsFailed = registry
			.counter(idFactory.name("requests").withTag("success", false));
	private static final Counter bytesUploaded = registry.counter(idFactory.name("upload-bytes"));
	private static final Counter bytesDownloaded = registry.counter(idFactory.name("download-bytes"));
	private static final Counter connectionAcquireWaits = registry.counter(idFactory.name("connection-acquire-waits"));
	private static final Counter connectionAcquireFailures = registry
			.counter(idFactory.name("connection-acquire-failures"));
	private static final Clock clock = registry.clock();

	public JdkHttpClient(NodePool nodePool, int maxConnections, Duration connectTimeout, Duration requestTimeout,
			Duration connectionAcquireTimeout, Map<String, String> defaultHeaders) {
		this.defaultHeaders = defaultHeaders;
		this.nodePool = nodePool;
		this.virtualThreadExecutor = Executors
				.newThreadPerTaskExecutor(Thread.ofVirtual().name("elasticsearch-client-request-", 0).factory());
		this.httpClient = HttpClient.newBuilder() //
				.version(HttpClient.Version.HTTP_1_1) //
				.connectTimeout(connectTimeout) //
				.executor(virtualThreadExecutor) //
				.build();
		this.connectionLimiter = new Semaphore(maxConnections);
		this.requestTimeout = requestTimeout;
		this.connectionAcquireTimeout = connectionAcquireTimeout;
	}

	private void acquireConnection() throws IOException {
		if (!connectionLimiter.tryAcquire()) {
			logger.debug("Connection limit reached, waiting for available slot...");
			long start = clock.monotonicTime();
			connectionAcquireWaits.increment();
			try {
				if (!connectionLimiter.tryAcquire(connectionAcquireTimeout.toMillis(), TimeUnit.MILLISECONDS)) {
					throw new IOException("Timeout waiting for available connection after "
							+ connectionAcquireTimeout.toMillis() + "ms");
				}
			} catch (InterruptedException e) {
				connectionAcquireFailures.increment();
				Thread.currentThread().interrupt();
				throw new IOException("Interrupted while waiting for connection", e);
			}
			connectionAcquireLatency.record(clock.monotonicTime() - start, TimeUnit.NANOSECONDS);
			logger.debug("Waited {}ms to acquire connection", Duration.ofNanos(System.nanoTime() - start).toMillis());
		}
	}

	@Override
	public TransportOptions createOptions(TransportOptions options) {
		return JdkHttpClientOptions.of(options);
	}

	@Override
	public Response performRequest(String endpointId, Node node, Request request, TransportOptions options)
			throws IOException {
		Node targetNode = node != null ? node : nodePool.next();
		acquireConnection();
		try {
			HttpRequest httpRequest = buildHttpRequest(targetNode, request, options);
			HttpResponse<InputStream> response = httpClient.send(httpRequest,
					HttpResponse.BodyHandlers.ofInputStream());
			trackResponseStatus(response);
			return new JdkResponse(response, targetNode, readResponseBody(response));
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			throw new IOException("Request interrupted", e);
		} finally {
			connectionLimiter.release();
		}
	}

	private CachedBinaryData readResponseBody(HttpResponse<InputStream> response) {
		InputStream inputStream = response.body();
		if (inputStream == null) {
			return null;
		}
		String contentType = response.headers().firstValue("Content-Type").orElse(null);
		byte[] bytes;
		try {
			bytes = inputStream.readAllBytes();
			bytesDownloaded.add(bytes.length);
		} catch (IOException e) {
			logger.error("IOException reading response", e);
			return null;
		}
		return new CachedBinaryData(bytes, contentType != null ? contentType : "application/json");
	}

	private void trackResponseStatus(HttpResponse<InputStream> response) {
		long statusCode = response.statusCode();
		if (statusCode >= 200 && statusCode < 300) {
			requestsSuccesses.increment();
		} else {
			requestsFailed.increment();
		}
	}

	@Override
	public CompletableFuture<Response> performRequestAsync(String endpointId, Node node, Request request,
			TransportOptions options) {
		Node targetNode = node != null ? node : nodePool.next();
		try {
			acquireConnection();
		} catch (IOException e) {
			return CompletableFuture.failedFuture(e);
		}
		try {
			HttpRequest httpRequest = buildHttpRequest(targetNode, request, options);
			return httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofInputStream()) //
					.thenApply(response -> {
						trackResponseStatus(response);
						return (Response) new JdkResponse(response, targetNode, readResponseBody(response));
					}).whenComplete((response, ex) -> {
						connectionLimiter.release();
						if (ex != null) {
							if (ex instanceof CompletionException cfe) {
								throw cfe;
							}
							throw new CompletionException(ex);
						}
					});
		} catch (IOException e) {
			connectionLimiter.release();
			return CompletableFuture.failedFuture(e);
		}
	}

	private HttpRequest buildHttpRequest(Node node, Request request, TransportOptions options) throws IOException {
		URI uri = buildUri(node, request);
		JdkHttpClientOptions jdkOptions = JdkHttpClientOptions.of(options);
		Duration timeout = jdkOptions.requestTimeout() != null ? jdkOptions.requestTimeout() : this.requestTimeout;
		HttpRequest.Builder builder = HttpRequest.newBuilder() //
				.uri(uri) //
				.timeout(timeout);

		HttpRequest.BodyPublisher bodyPublisher = buildBodyPublisher(request);
		builder.method(request.method(), bodyPublisher);

		for (Map.Entry<String, String> header : defaultHeaders.entrySet()) {
			builder.header(header.getKey(), header.getValue());
		}
		for (Map.Entry<String, String> header : request.headers().entrySet()) {
			builder.header(header.getKey(), header.getValue());
		}
		for (Map.Entry<String, String> header : jdkOptions.headers()) {
			builder.header(header.getKey(), header.getValue());
		}
		return builder.build();
	}

	private URI buildUri(Node node, Request request) {
		StringBuilder sb = new StringBuilder(node.uri().toString());

		// Ensure no double slash between base URL and path
		String path = request.path();
		if (sb.charAt(sb.length() - 1) == '/' && path.startsWith("/")) {
			sb.setLength(sb.length() - 1);
		}
		sb.append(path);

		Map<String, String> queryParams = request.queryParams();
		if (queryParams != null && !queryParams.isEmpty()) {
			boolean first = !path.contains("?");
			for (Map.Entry<String, String> param : queryParams.entrySet()) {
				sb.append(first ? '?' : '&');
				first = false;
				sb.append(encodeUriComponent(param.getKey()));
				sb.append('=');
				sb.append(encodeUriComponent(param.getValue()));
			}
		}

		return URI.create(sb.toString());
	}

	private String encodeUriComponent(String value) {
		return URLEncoder.encode(value, java.nio.charset.StandardCharsets.UTF_8);
	}

	private HttpRequest.BodyPublisher buildBodyPublisher(Request request) throws IOException {
		Iterable<ByteBuffer> body = request.body();
		if (body == null) {
			return HttpRequest.BodyPublishers.noBody();
		}

		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		for (ByteBuffer buffer : body) {
			ByteBuffer dup = buffer.duplicate();
			byte[] bytes = new byte[dup.remaining()];
			dup.get(bytes);
			baos.write(bytes);
		}
		bytesUploaded.add(baos.size());
		return HttpRequest.BodyPublishers.ofByteArray(baos.toByteArray());
	}

	@Override
	public void close() throws IOException {
		virtualThreadExecutor.close();
	}

	private static class JdkResponse implements Response {
		private final HttpResponse<InputStream> httpResponse;
		private final Node node;
		private final CachedBinaryData bodyData;
		private volatile boolean closed;

		JdkResponse(HttpResponse<InputStream> httpResponse, Node node, CachedBinaryData bodyData) {
			this.httpResponse = httpResponse;
			this.node = node;
			this.bodyData = bodyData;
		}

		@Override
		public Node node() {
			return node;
		}

		@Override
		public int statusCode() {
			return httpResponse.statusCode();
		}

		@Override
		public String header(String name) {
			return httpResponse.headers().firstValue(name).orElse(null);
		}

		@Override
		public List<String> headers(String name) {
			List<String> values = httpResponse.headers().allValues(name);
			return values.isEmpty() ? Collections.emptyList() : values;
		}

		@Override
		public BinaryData body() throws IOException {
			return bodyData;
		}

		@Override
		public Object originalResponse() {
			return httpResponse;
		}

		@Override
		public void close() throws IOException {
			if (!closed) {
				closed = true;
				httpResponse.body().close();
			}
		}
	}

	private record CachedBinaryData(byte[] data, String contentType) implements BinaryData {
		@Override
		public void writeTo(OutputStream out) throws IOException {
			out.write(data);
		}

		@Override
		public ByteBuffer asByteBuffer() {
			return ByteBuffer.wrap(data);
		}

		@Override
		public InputStream asInputStream() {
			return new ByteArrayInputStream(data);
		}

		@Override
		public boolean isRepeatable() {
			return true;
		}

		@Override
		public long size() {
			return data.length;
		}
	}
}
