package net.bluemind.lib.elasticsearch;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import co.elastic.clients.elasticsearch._helpers.bulk.BulkIngester;
import co.elastic.clients.elasticsearch._helpers.bulk.BulkListener;
import co.elastic.clients.elasticsearch.core.BulkRequest;
import co.elastic.clients.elasticsearch.core.BulkResponse;
import co.elastic.clients.elasticsearch.core.bulk.BulkOperation;
import co.elastic.clients.elasticsearch.core.bulk.BulkResponseItem;
import co.elastic.clients.util.ObjectBuilder;
import net.bluemind.lib.elasticsearch.exception.ElasticBulkException;

public class EsBulk {
	private static final Logger logger = LoggerFactory.getLogger(EsBulk.class);
	private static final int MAX_OPERATIONS = -1; // Configurable bulk size limit, -1 : Unlimited
	private static final long MAX_SIZE = (long) (10 * Math.pow(1024, 2));
	// 5MB, no operations limit, conservative guess based on ElasticSearch's
	// recommendation
	// https://www.elastic.co/guide/en/elasticsearch/guide/current/indexing-performance.html#_using_and_sizing_bulk_requests
	private static final long FLUSH_INTERVAL_MS = 1000;

	private final ElasticsearchClient esClient;

	public EsBulk(ElasticsearchClient esClient) {
		this.esClient = esClient;
	}

	public <T> Optional<BulkResponse> commitAll(List<T> toBulk,
			BiFunction<T, BulkOperation.Builder, ObjectBuilder<BulkOperation>> map) {
		return commitAll(toBulk, map, 1, Duration.ofMinutes(10));
	}

	public <T> Optional<BulkResponse> commitAll(List<T> toBulk,
			BiFunction<T, BulkOperation.Builder, ObjectBuilder<BulkOperation>> map, int concurrentRequests,
			Duration timeout) {
		if (toBulk.isEmpty()) {
			logger.info("Empty bulk, not running.");
			return Optional.empty();
		}

		CompletableFuture<BulkResponse> futureResponse = new CompletableFuture<>();

		try (BulkIngester<T> ingester = BulkIngester.of(b -> b.client(esClient) //
				.maxOperations(MAX_OPERATIONS) //
				.maxConcurrentRequests(concurrentRequests) //
				.maxSize(MAX_SIZE) //

				.flushInterval(FLUSH_INTERVAL_MS, TimeUnit.MILLISECONDS) //
				.listener(new BulkListener<T>() {
					@Override
					public void beforeBulk(long executionId, BulkRequest request, List<T> contexts) {
						logger.debug("Starting bulk request {}", executionId);
					}

					@Override
					public void afterBulk(long executionId, BulkRequest request, List<T> contexts,
							BulkResponse response) {
						reportErrors(response);
						futureResponse.complete(response);
					}

					@Override
					public void afterBulk(long executionId, BulkRequest request, List<T> contexts, Throwable failure) {
						logger.error("Bulk request {} failed", executionId, failure);
						futureResponse.completeExceptionally(new ElasticBulkException(failure));
					}
				}))) {

			for (T item : toBulk) {
				BulkOperation operation = map.apply(item, new BulkOperation.Builder()).build();
				ingester.add(operation);
				// Add Operations to the ingester.
				// Each time we reach one of these conditions, a request is queued :
				// - MAX_SIZE (checked on ingester.add())
				// - MAX_OPERATIONS (checked on ingester.add())
				// - FLUSH_INTERVAL_MS (separate thread flushing at the configured interval)
			}

			ingester.flush(); // Avoid waiting for the FLUSH_INTERVAL_MS

			// In case of a big bulk, multiple requests might be needed.
			// Requests are sent one after the other.
			// so even the timeout for one request is 30s,
			// we might need more time to complete all of them.
			BulkResponse response = futureResponse.get(timeout.toSeconds(), TimeUnit.SECONDS);
			return Optional.of(response);

		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			throw new ElasticBulkException(e);
		} catch (Exception e) {
			throw new ElasticBulkException(e);
		}
	}

	private void reportErrors(BulkResponse response) {
		if (!response.errors()) {
			return;
		}
		List<BulkResponseItem> failedItems = response.items().stream().filter(i -> i.error() != null).toList();
		failedItems.forEach(i -> logger.error("Bulk request failed on index:{} id:{} error:{} stack:{}", i.index(),
				i.id(), i.error().type(), i.error().stackTrace()));
	}
}