/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keydb.sessions.codec;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.async.ByteBufferFeeder;

import io.lettuce.core.codec.RedisCodec;
import io.lettuce.core.codec.StringCodec;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.Unpooled;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.context.SecurityContext;

public class SecurityContextCodec implements RedisCodec<String, SecurityContext> {

	private static final Logger logger = LoggerFactory.getLogger(SecurityContextCodec.class);
	public static final int CODEC_VERSION = 1;

	private final JsonFactory jsonFactory;

	public SecurityContextCodec() {
		this.jsonFactory = new JsonFactory();
	}

	@Override
	public String decodeKey(ByteBuffer bytes) {
		return StringCodec.ASCII.decodeKey(bytes);
	}

	@Override
	public SecurityContext decodeValue(ByteBuffer bytes) {

		try (JsonParser parser = jsonFactory.createNonBlockingByteBufferParser()) {
			ByteBufferFeeder feeder = (ByteBufferFeeder) parser.getNonBlockingInputFeeder();
			feeder.feedInput(bytes);
			feeder.endOfInput();

			return pullParse(parser);

		} catch (IOException e) {
			logger.error(e.getMessage(), e);
			return null;
		}
	}

	private static class SecCtxBuilder {
		long created = 0L;
		String sessionId = null;
		String subject = null;
		String subjectDn = null;
		String domainUid = null;
		String lang = null;
		String origin = null;
		boolean isInteractive = false;
		String ownerPrincipal = null;
		List<String> memberOf = new ArrayList<>();
		List<String> roles = new ArrayList<>();
		List<String> remoteAddresses = new ArrayList<>();
		Map<String, Set<String>> orgUnitRoles = new HashMap<>();
		Long validityMs = null;

		public SecurityContext build() {
			return new SecurityContext(created, sessionId, subject, subjectDn, memberOf, roles, orgUnitRoles, domainUid,
					lang, origin, isInteractive, ownerPrincipal, remoteAddresses, validityMs);
		}

	}

	private interface IFieldMapper {
		public static IFieldMapper NOOP = (s, p) -> {
		};

		void handle(SecCtxBuilder toBuild, JsonParser parser) throws IOException;
	}

	private static final Map<String, IFieldMapper> HANDLERS = setupParseHandlers();

	private SecurityContext pullParse(JsonParser parser) throws IOException {
		SecCtxBuilder builder = new SecCtxBuilder();

		String currentFieldName = null;
		JsonToken tok;
		while ((tok = parser.nextToken()) != JsonToken.END_OBJECT && tok != null) {
			if (parser.getCurrentToken() == JsonToken.FIELD_NAME) {
				currentFieldName = parser.currentName();
				continue;
			}
			HANDLERS.getOrDefault(currentFieldName, IFieldMapper.NOOP).handle(builder, parser);
		}
		return builder.build();
	}

	private static Map<String, IFieldMapper> setupParseHandlers() {
		Map<String, IFieldMapper> handlers = new HashMap<>();

		handlers.put("created", (toBuild, parser) -> toBuild.created = parser.getLongValue());
		handlers.put("sessionId", (toBuild, parser) -> toBuild.sessionId = parser.getValueAsString());
		handlers.put("subject", (toBuild, parser) -> toBuild.subject = parser.getValueAsString());
		handlers.put("subjectDisplayName", (toBuild, parser) -> toBuild.subjectDn = parser.getValueAsString());
		handlers.put("domainUid", (toBuild, parser) -> toBuild.domainUid = parser.getValueAsString());
		handlers.put("lang", (toBuild, parser) -> toBuild.lang = parser.getValueAsString());
		handlers.put("origin", (toBuild, parser) -> toBuild.origin = parser.getValueAsString());
		handlers.put("interactive", (toBuild, parser) -> toBuild.isInteractive = parser.getValueAsBoolean());
		handlers.put("ownerPrincipal", (toBuild, parser) -> toBuild.ownerPrincipal = parser.getValueAsString());
		handlers.put("validityMs", (toBuild, parser) -> toBuild.validityMs = parser.getLongValue());

		handlers.put("memberOf", (toBuild, parser) -> fillList(toBuild.memberOf, parser));
		handlers.put("roles", (toBuild, parser) -> fillList(toBuild.roles, parser));
		handlers.put("remoteAddresses", (toBuild, parser) -> fillList(toBuild.remoteAddresses, parser));

		handlers.put("orgUnitsRoles", (toBuild, parser) -> fillMap(toBuild.orgUnitRoles, parser));
		return handlers;
	}

	private static void fillMap(Map<String, Set<String>> toFill, JsonParser parser) throws IOException {
		JsonToken tok;
		if (parser.getCurrentToken() == JsonToken.START_OBJECT) {
			while ((tok = parser.nextToken()) != JsonToken.END_OBJECT && tok != null) {
				if (parser.getCurrentToken() == JsonToken.FIELD_NAME) {
					String ouId = parser.currentName();
					parser.nextToken();
					Set<String> ouRoles = new LinkedHashSet<>();
					if (parser.getCurrentToken() == JsonToken.START_ARRAY) {
						while ((tok = parser.nextToken()) != JsonToken.END_ARRAY && tok != null) {
							ouRoles.add(parser.getValueAsString());
						}
					}
					toFill.put(ouId, ouRoles);
				}
			}
		}
	}

	private static void fillList(List<String> toFill, JsonParser parser) throws IOException {
		JsonToken tok;
		if (parser.getCurrentToken() == JsonToken.START_ARRAY) {
			while ((tok = parser.nextToken()) != JsonToken.END_ARRAY && tok != null) {
				String elem = parser.getValueAsString();
				toFill.add(elem);
			}
		}
	}

	@Override
	public ByteBuffer encodeKey(String key) {
		return StringCodec.ASCII.encodeKey(key);
	}

	@Override
	public ByteBuffer encodeValue(SecurityContext value) {
		return toJson(value);
	}

	private void writeArray(JsonGenerator gen, String arrayName, Collection<String> values) throws IOException {
		gen.writeFieldName(arrayName);
		gen.writeStartArray();
		for (String v : values) {
			gen.writeString(v);
		}
		gen.writeEndArray();
	}

	private ByteBuffer toJson(SecurityContext sc) {
		ByteBuf buf = Unpooled.buffer();
		try (OutputStream out = new ByteBufOutputStream(buf); JsonGenerator gen = jsonFactory.createGenerator(out)) {
			gen.writeStartObject();
			gen.writeNumberField("created", sc.getCreated());
			gen.writeStringField("sessionId", sc.getSessionId());
			gen.writeStringField("subject", sc.getSubject());
			gen.writeStringField("subjectDisplayName", sc.getSubjectDisplayName());
			gen.writeStringField("domainUid", sc.getContainerUid());
			gen.writeStringField("lang", sc.getLang());
			gen.writeStringField("origin", sc.getOrigin());
			gen.writeBooleanField("interactive", sc.isInteractive());
			gen.writeStringField("ownerPrincipal", sc.getOwnerPrincipal());
			if (sc.getValidityPeriodMs() != null) {
				gen.writeNumberField("validityMs", sc.getValidityPeriodMs());
			}

			writeArray(gen, "memberOf", sc.getMemberOf());
			writeArray(gen, "roles", sc.getRoles());
			writeArray(gen, "remoteAddresses", sc.getRemoteAddresses());

			gen.writeFieldName("orgUnitsRoles");
			gen.writeStartObject();
			for (Entry<String, Set<String>> entry : sc.getRolesByOrgUnits().entrySet()) {
				writeArray(gen, entry.getKey(), entry.getValue());
			}
			gen.writeEndObject();

			gen.writeEndObject();
		} catch (IOException e) {
			e.printStackTrace();
			throw new ServerFault(e);
		}
		return buf.nioBuffer();
	}

}
