/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
*/
package net.bluemind.keydb.sessions;

import java.net.SocketAddress;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import io.lettuce.core.RedisChannelHandler;
import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisConnectionStateListener;
import io.lettuce.core.ScanArgs;
import io.lettuce.core.ScanIterator;
import io.lettuce.core.SetArgs;
import io.lettuce.core.api.async.RedisAsyncCommands;
import io.lettuce.core.api.push.PushMessage;
import io.lettuce.core.api.sync.RedisCommands;
import io.lettuce.core.codec.StringCodec;
import io.lettuce.core.pubsub.StatefulRedisPubSubConnection;
import io.lettuce.core.pubsub.api.async.RedisPubSubAsyncCommands;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.vertx.core.Context;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.sessions.ISessionDeletionListener;
import net.bluemind.core.sessions.ISessionsStore;
import net.bluemind.core.sessions.SessionDeletionListeners;
import net.bluemind.keydb.common.ClientProvider;
import net.bluemind.keydb.common.KeydbBootstrapNetAddress;
import net.bluemind.keydb.sessions.codec.SecurityContextCodec;
import net.bluemind.lib.vertx.VertxContext;
import net.bluemind.lib.vertx.VertxPlatform;

public class SecurityContextStore implements ISessionsStore {

	private static final Logger logger = LoggerFactory.getLogger(SecurityContextStore.class);
	private static final String IDENTITY = UUID.randomUUID().toString();

	private static final ExecutorService executorService = Executors
			.newSingleThreadExecutor(new DefaultThreadFactory("security-context-store-delete-session"));

	private final Duration expireDuration;

	private RedisPubSubAsyncCommands<String, String> expireCommands;
	private RedisCommands<String, SecurityContext> valueCommands;
	private RedisAsyncCommands<String, SecurityContext> valueAsyncCommands;

	public static final Cache<String, SecurityContext> fastCache = Caffeine.newBuilder().recordStats()
			.expireAfterAccess(30, TimeUnit.SECONDS) //
			.maximumSize(65535).build();
	public static final Cache<String, Boolean> expireDedup = Caffeine.newBuilder().recordStats()
			.expireAfterWrite(10, TimeUnit.SECONDS) //
			.maximumSize(65535).build();

	public static class CacheRegistration implements ICacheRegistration {
		@Override
		public void registerCaches(CacheRegistry cr) {
			cr.register(SecurityContextStore.class, fastCache);
			cr.register(SessionsHelper.class, expireDedup);

		}
	}

	public SecurityContextStore() {
		expireDuration = CoreConfig.get().getDuration(CoreConfig.Sessions.IDLE_TIMEOUT);

		RedisClient redisClient = ClientProvider.newClient();
		StatefulRedisPubSubConnection<String, String> pubSubAsync = redisClient.connectPubSub();
		expireCommands = pubSubAsync.async();

		// For expiration events
		expireCommands.configSet("notify-keyspace-events", ClientProvider.NOTIFY_KEYSPACE_EVENTCONFIG);
		expireCommands.subscribe("__keyevent@0__:expired");
		initConnectionListener(redisClient, expireCommands);
		initExpirationListener(pubSubAsync);
		var redisConnection = redisClient.connect(new SecurityContextCodec());
		valueCommands = redisConnection.sync();
		valueAsyncCommands = redisConnection.async();
		logger.info("Keydb connection setup completed ({})", KeydbBootstrapNetAddress.getKeydbIP());
	}

	private void initConnectionListener(RedisClient redisClient, RedisPubSubAsyncCommands<String, String> commands) {
		redisClient.addListener(new RedisConnectionStateListener() {
			@Override
			public void onRedisConnected(RedisChannelHandler<?, ?> handlerConnection, SocketAddress socketAddress) {
				Thread.ofVirtual().start(() -> {
					commands.configSet("notify-keyspace-events", ClientProvider.NOTIFY_KEYSPACE_EVENTCONFIG);
					logger.debug("[keydb] Config set on notify keyspace events");
				});
			}
		});
	}

	private void initExpirationListener(StatefulRedisPubSubConnection<String, String> pubSubAsync) {
		pubSubAsync.addListener((PushMessage message) -> {
			executorService.submit(() -> {
				try {
					String key = (String) message.getContent(StringCodec.ASCII::decodeKey).get(2);
					String sid = SessionsHelper.sidFromExpirationKey(key);
					String payloadKey = SessionsHelper.valueHolder(sid);
					fastCache.invalidate(payloadKey);
					SecurityContext ctx = valueCommands.get(payloadKey);

					if (ctx != null) {
						notifySessionRemovalListeners(sid, ctx);
						valueCommands.del(payloadKey);
					}
				} catch (ClassCastException e) {
					// If we catch message of type "subscribe" on expired events
					logger.debug("[keydb] Subscribed to expired events");
				}
			});
		});
	}

	@SuppressWarnings("serial")
	public static class KeyDbConnectionNotAvalaible extends RuntimeException {
		KeyDbConnectionNotAvalaible(Throwable t) {
			super("NO: KeydbSessionsClient is not available: ", t);
		}
	}

	private Context getVertxContext() {
		return VertxContext.getOrCreateDuplicatedContext(VertxPlatform.getVertx());
	}

	public void notifySessionRemovalListeners(String sessionId, SecurityContext securityContext) {
		for (ISessionDeletionListener listener : SessionDeletionListeners.get()) {
			getVertxContext().executeBlocking(() -> {
				listener.deleted(IDENTITY, sessionId, securityContext);
				return null;
			}, true).andThen(asyncResult -> {
				if (!asyncResult.succeeded()) {
					asyncResult.cause();
				}
			});
		}
	}

	private Duration getExpireDuration(SecurityContext ctx) {
		return Optional.ofNullable(ctx.getValidityPeriodMs()).map(expireMs -> Duration.ofMillis(expireMs))
				.orElse(expireDuration);
	}

	@Override
	public SecurityContext getIfPresent(String sid) {
		SecurityContext ctx = fastCache.get(SessionsHelper.valueHolder(sid), valueKey -> valueCommands.get(valueKey));
		if (ctx != null) {
			expireDedup.get(SessionsHelper.expirationKey(sid), expireKey -> {
				expireCommands.expire(expireKey, getExpireDuration(ctx));
				return true;
			});
		}
		return ctx;
	}

	@Override
	public void put(String sid, SecurityContext value) {
		String valueKey = SessionsHelper.valueHolder(sid);
		valueAsyncCommands.set(valueKey, value);
		expireCommands.set(SessionsHelper.expirationKey(sid), "", SetArgs.Builder.ex(getExpireDuration(value)));
		fastCache.put(valueKey, value);
	}

	@Override
	public void invalidate(String sid) {
		logger.info("Invalidate session {}", sid);
		SecurityContext valueSession = getIfPresent(sid);
		String valueKey = SessionsHelper.valueHolder(sid);
		String expireKey = SessionsHelper.valueHolder(sid);
		fastCache.invalidate(valueKey);
		expireDedup.invalidate(expireKey);
		valueAsyncCommands.del(valueKey);
		valueAsyncCommands.del(expireKey);
		notifySessionRemovalListeners(sid, valueSession);
	}

	@Override
	public Map<String, SecurityContext> asMap() {
		ScanIterator<String> scan = ScanIterator.scan(valueCommands,
				ScanArgs.Builder.matches(SessionsHelper.CORE_VALUE_HOLDER + "*"));
		Map<String, SecurityContext> map = new HashMap<>();
		while (scan.hasNext()) {
			String sid = SessionsHelper.sidFromValueHolder(scan.next());
			map.put(sid, this.getIfPresent(sid));
		}
		return map;
	}

	@Override
	public void invalidateAll() {
		fastCache.invalidateAll();
		ScanIterator<String> scan = ScanIterator.scan(valueCommands,
				ScanArgs.Builder.matches(SessionsHelper.CORE_VALUE_HOLDER + "*"));
		while (scan.hasNext()) {
			String sid = scan.next();
			valueAsyncCommands.del(sid);
			valueAsyncCommands.del(SessionsHelper.expirationKey(SessionsHelper.sidFromValueHolder(sid)));
		}
	}
}
