/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
*/
package net.bluemind.keydb.sessions;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.lettuce.core.RedisClient;
import io.lettuce.core.ScanArgs;
import io.lettuce.core.ScanIterator;
import io.lettuce.core.api.sync.RedisCommands;
import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import net.bluemind.core.sessions.SessionsStoreFactory;
import net.bluemind.keydb.common.ClientProvider;
import net.bluemind.keydb.sessiondata.SessionDataStore;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.lib.vertx.VertxPlatform;

public class CleanValueHolderVerticle extends AbstractVerticle {
	private static final Logger logger = LoggerFactory.getLogger(CleanValueHolderVerticle.class);

	private final long nbMilliSecondsTimer;

	public CleanValueHolderVerticle() {
		this(TimeUnit.MINUTES.toMillis(2));
	}

	public CleanValueHolderVerticle(long nbMilliSecondsTimer) {
		this.nbMilliSecondsTimer = nbMilliSecondsTimer;
	}

	@Override
	public void start() throws Exception {
		vertx.setTimer(nbMilliSecondsTimer, ev -> {
			this.execute(ev);
			VertxPlatform.executeBlockingPeriodic(vertx, TimeUnit.HOURS.toMillis(2), this::execute);
		});
	}

	private void execute(Long timerId) {
		try (RedisClient keydbCli = connect()) {
			RedisCommands<String, String> commands = keydbCli.connect().sync();

			Set<String> valueHolderSids = scan(commands, SessionsHelper.CORE_VALUE_HOLDER).stream()
					.map(SessionsHelper::sidFromValueHolder).collect(Collectors.toSet());
			Set<String> expirationSids = scan(commands, SessionsHelper.CORE_EXPIRATION_KEY).stream()
					.map(SessionsHelper::sidFromExpirationKey).collect(Collectors.toSet());
			Set<String> dataSids = scan(commands, SessionDataStore.DATA_VALUE_HOLDER).stream()
					.map(SessionsHelper::sidFromDataHolder).collect(Collectors.toSet());

			Set<String> orphanValueSid = valueHolderSids.stream().filter(valueSid -> !expirationSids.contains(valueSid))
					.collect(Collectors.toSet());

			Set<String> orphanDataSid = dataSids.stream().filter(orphanValueSid::contains).collect(Collectors.toSet());

			// clean orphanValueSid UNION orphanDataSid
			logger.info("Cleaning session value holders without expiration pair (tid {})", timerId);
			orphanDataSid.stream().forEach(sid -> {
				SessionsStoreFactory.get().invalidate(sid);
			});

			// clean orphanValueSid EXCEPT orphanDataSid
			// security context cannot be rebuild without dataValue so we can't execute
			// ISessionDeletionListener, so we don't call invalidate
			Set<String> orphanValueKeysNotInOrphanDataSid = orphanValueSid.stream()
					.filter(sid -> !orphanDataSid.contains(sid)).map(SessionsHelper::valueHolder)
					.collect(Collectors.toSet());
			removeAll(commands, orphanValueKeysNotInOrphanDataSid);

			// clean orphanDataSid EXCEPT orphanValueSid is a normal case, nothing to clean
			// invalidate function delete value key, but keep data key to execute
			// ISessionDeletionListener
		}
	}

	private Set<String> scan(RedisCommands<String, String> commands, String match) {
		Set<String> sidsToMatch = new HashSet<>();
		ScanIterator<String> scan = ScanIterator.scan(commands, ScanArgs.Builder.matches(match + "*"));
		while (scan.hasNext()) {
			sidsToMatch.add(scan.next());
		}
		return sidsToMatch;
	}

	private void removeAll(RedisCommands<String, String> commands, Set<String> keysToRemove) {
		keysToRemove.forEach(commands::del);
	}

	private RedisClient connect() {
		return ClientProvider.newClient();
	}

	public static final class Factory implements IVerticleFactory, IUniqueVerticleFactory {
		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new CleanValueHolderVerticle();
		}
	}
}