/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.internal;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.http.HttpMethod;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.api.OidcClient;
import net.bluemind.keycloak.utils.KeycloakAdminClient;
import net.bluemind.keycloak.utils.adapters.OidcClientAdapter;
import net.bluemind.keycloak.utils.endpoint.KeycloakEndpoints;

public class OidcService {
	private static final Logger logger = LoggerFactory.getLogger(OidcService.class);

	private final String realmId;

	public OidcService(String realmId) {
		this.realmId = realmId;
	}

	public void create(String domainUid, String clientId) {
		logger.info("Domain: {} - Realm {}: Create client {}", domainUid, realmId, clientId);

		// List flows to get our flow Id
		String ourFlowId = null;
		JsonObject resp = KeycloakAdminClient.getInstance().call(KeycloakEndpoints.flowsEndpoint(realmId),
				HttpMethod.GET);
		JsonArray flows = resp.getJsonArray("results");
		for (int i = 0; i < flows.size(); i++) {
			JsonObject curFlow = flows.getJsonObject(i);
			if (IKeycloakUids.BLUEMIND_FLOW_ALIAS.equals(curFlow.getString("alias"))) {
				ourFlowId = curFlow.getString("id");
			}
		}

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance().execute(
				KeycloakEndpoints.clientsEndpoint(realmId), HttpMethod.POST,
				OidcClientAdapter.build(domainUid, clientId, Optional.ofNullable(ourFlowId)).toJson());

		try {
			JsonObject json = response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
			if (json.containsKey("error")) {
				throw new ServerFault("Error: " + json.getString("error") + " - " + json.getString("error_description"),
						ErrorCode.UNKNOWN);
			}
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			throw new ServerFault("Failed to create client", e);
		}
	}

	public String getSecret(String clientId) {
		logger.info("Realm {}: Get client secret {}", realmId, clientId);
		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance()
				.execute(KeycloakEndpoints.clientCredsEndpoint(realmId, clientId), HttpMethod.GET);

		JsonObject json;
		try {
			json = response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			throw new ServerFault("Failed to get client secret", e);
		}

		if (json == null) {
			logger.warn("Failed to fetch secret");
			return null;
		}

		return json.getString("value");
	}

	public List<OidcClient> allOidClients() {
		logger.info("Realm {}: Get OIDC clients", realmId);

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance()
				.execute(KeycloakEndpoints.clientsEndpoint(realmId), HttpMethod.GET);
		JsonObject json;
		try {
			json = response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			throw new ServerFault("Failed to fetch clients for realm " + realmId, e);
		}

		List<OidcClient> ret = new ArrayList<>();
		JsonArray results = json.getJsonArray("results");
		results.forEach(cli -> {
			if (cli != null && "openid-connect".equals(((JsonObject) cli).getString("protocol"))) {
				ret.add(OidcClientAdapter.fromJson((JsonObject) cli).oidcClient);
			}
		});

		return ret;
	}

	public OidcClient getOidcClient(String clientId) {
		logger.info("Realm {}: Get client {}", realmId, clientId);
		String spec = KeycloakEndpoints.clientsEndpoint(realmId) + "?clientId="
				+ URLEncoder.encode(clientId, StandardCharsets.UTF_8);

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance().execute(spec, HttpMethod.GET);
		JsonObject json;
		try {
			json = response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			logger.error("Exception " + e.getClass().getName() + " : " + e.getMessage(), e);
			throw new ServerFault("Failed to fetch client " + clientId + " for realm " + realmId);
		}

		if (json == null) {
			logger.warn("Failed to fetch client id {}", clientId);
			return null;
		}

		JsonArray results = json.getJsonArray("results");
		if (results == null || results.size() == 0
				|| !"openid-connect".equals(json.getJsonArray("results").getJsonObject(0).getString("protocol"))) {
			return null;
		}

		return OidcClientAdapter.fromJson(json.getJsonArray("results").getJsonObject(0)).oidcClient;
	}

	public void deleteOidcClient(String clientId) {
		logger.info("Realm {}: Delete client {}", realmId, clientId);

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance()
				.execute(KeycloakEndpoints.clientsEndpoint(realmId) + "/" + clientId, HttpMethod.DELETE);
		try {
			response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			throw new ServerFault("Failed to delete client", e);
		}
	}

	public void updateClient(String clientId, OidcClient oc) {
		logger.info("Realm {}: Update client {}", realmId, clientId);

		String oidcClientId = oc.id;
		if (oidcClientId == null) {
			OidcClient cli = getOidcClient(clientId);

			if (cli == null) {
				throw new ServerFault(clientId + " not found in realm " + realmId + " - to update it");
			}

			oidcClientId = cli.id;
		}

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance().execute(
				KeycloakEndpoints.clientsEndpoint(realmId) + "/" + oidcClientId, HttpMethod.PUT,
				new OidcClientAdapter(oc).toJson());
		try {
			response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			throw new ServerFault("Failed to update client " + clientId, e);
		}
	}
}
