/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.internal;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.keycloak.api.BluemindProviderComponent;
import net.bluemind.keycloak.api.IKeycloakBluemindProviderAdmin;
import net.bluemind.keycloak.internal.ComponentService.ComponentProvider;
import net.bluemind.keycloak.utils.adapters.BlueMindComponentAdapter;
import net.bluemind.role.api.BasicRoles;

public class KeycloakBluemindProviderAdminService implements IKeycloakBluemindProviderAdmin {
	private static final Logger logger = LoggerFactory.getLogger(KeycloakBluemindProviderAdminService.class);

	private final RBACManager rbacManager;
	private final ComponentService componentService;
	private final String domainUid;

	public KeycloakBluemindProviderAdminService(BmContext context, String domainUid) {
		this.rbacManager = new RBACManager(context);
		this.componentService = ComponentService.forDomain(domainUid);
		this.domainUid = domainUid;
	}

	@Override
	public void create(BluemindProviderComponent component) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		componentService.createComponent(new BlueMindComponentAdapter(component).toJson());
	}

	@Override
	public List<BluemindProviderComponent> allBluemindProviders() throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		logger.info("Domain {}: Get all Bluemind providers", domainUid);

		List<BluemindProviderComponent> ret = new ArrayList<>();
		componentService.allComponents(ComponentProvider.BLUEMIND)
				.forEach(cmp -> BlueMindComponentAdapter.fromJson(cmp).ifPresent(ret::add));
		return ret;
	}

	@Override
	public BluemindProviderComponent getBluemindProvider(String componentName) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		logger.info("Domain {}: Get Bluemind provider {}", domainUid, componentName);

		return BlueMindComponentAdapter
				.fromJson(componentService.getComponent(ComponentProvider.BLUEMIND, componentName)).orElse(null);
	}

	@Override
	public void deleteBluemindProvider(String componentName) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		logger.info("Domain {}: Delete bluemind provider {}", domainUid, componentName);

		componentService.deleteComponent(ComponentProvider.BLUEMIND, componentName);
	}
}
