/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.internal;

import java.util.List;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.keycloak.api.IKeycloakClientAdmin;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.api.OidcClient;
import net.bluemind.role.api.BasicRoles;

public class KeycloakClientAdminService implements IKeycloakClientAdmin {
	private RBACManager rbacManager;
	private final OidcService oidcService;
	private final String domainUid;

	public KeycloakClientAdminService(BmContext context, String domainUid) {
		this.rbacManager = new RBACManager(context);
		this.oidcService = new OidcService(IKeycloakUids.realmId(domainUid));
		this.domainUid = domainUid;
	}

	@Override
	public void create(String clientId) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		oidcService.create(domainUid, clientId);
	}

	@Override
	public String getSecret(String clientId) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		return oidcService.getSecret(clientId);
	}

	@Override
	public List<OidcClient> allOidcClients() throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		return oidcService.allOidClients();
	}

	@Override
	public OidcClient getOidcClient(String clientId) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		return oidcService.getOidcClient(clientId);
	}

	@Override
	public void deleteOidcClient(String clientId) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		oidcService.deleteOidcClient(clientId);
	}

	@Override
	public void updateClient(String clientId, OidcClient oc) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		oidcService.updateClient(clientId, oc);
	}

}