/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.internal;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;

import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.auth.AuthDomainProperties;
import net.bluemind.core.api.auth.AuthTypes;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.keycloak.api.IKeycloakAdmin;
import net.bluemind.keycloak.api.IKeycloakBluemindProviderAdmin;
import net.bluemind.keycloak.api.IKeycloakClientAdmin;
import net.bluemind.keycloak.api.IKeycloakFlowAdmin;
import net.bluemind.keycloak.api.IKeycloakKerberosAdmin;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.api.KerberosComponent;
import net.bluemind.keycloak.api.OidcClient;
import net.bluemind.keycloak.utils.KerberosConfigHelper;
import net.bluemind.keycloak.utils.KeycloakUrls;
import net.bluemind.keycloak.utils.adapters.BlueMindComponentAdapter;
import net.bluemind.system.api.IInternalCredentials;
import net.bluemind.utils.SyncHttpClient;

public class KeycloakManager {
	private static final Logger logger = LoggerFactory.getLogger(KeycloakManager.class);
	private static final int KEYCLOAK_WAIT_MAX_RETRIES = 8; // 5sec per retry => 40sec max wait
	private static final String GLOBAL_VIRT = "global.virt";

	private final ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
	private final ItemValue<Domain> domain;

	private KeycloakManager(ItemValue<Domain> domain) {
		this.domain = domain;
	}

	public static KeycloakManager forDomain(String domainUid) {
		ItemValue<Domain> domain = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IInCoreDomains.class).getUnfiltered(domainUid);

		if (domain == null || domain.value == null) {
			throw ServerFault.notFound("Domain " + domainUid + " not found");
		}

		return new KeycloakManager(domain);
	}

	private void initKeycloak() {
		logger.info("Init Keycloak for domain {}", domain.uid);

		IKeycloakAdmin keycloakAdminService = provider.instance(IKeycloakAdmin.class);

		String clientId = IKeycloakUids.clientId(IKeycloakUids.realmId(domain.uid));

		keycloakAdminService.createRealm(domain.uid);

		IKeycloakFlowAdmin keycloakFlowService = provider.instance(IKeycloakFlowAdmin.class, domain.uid);
		keycloakFlowService.createByCopying(IKeycloakUids.KEYCLOAK_FLOW_ALIAS, IKeycloakUids.BLUEMIND_FLOW_ALIAS);

		IKeycloakBluemindProviderAdmin keycloakBluemindProviderService = provider
				.instance(IKeycloakBluemindProviderAdmin.class, domain.uid);
		keycloakBluemindProviderService.create(BlueMindComponentAdapter.build(domain.uid).component);

		String authType = domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name());
		if (Strings.isNullOrEmpty(authType)) {
			authType = AuthTypes.INTERNAL.name();
			domain.value.properties.put(AuthDomainProperties.AUTH_TYPE.name(), authType);
		}

		IKeycloakClientAdmin keycloakClientAdmin = provider.instance(IKeycloakClientAdmin.class, domain.uid);
		keycloakClientAdmin.create(clientId);
		String secret = keycloakClientAdmin.getSecret(clientId);
		Map<String, String> properties = domain.value.properties != null ? domain.value.properties : new HashMap<>();
		properties.put(AuthDomainProperties.OPENID_CLIENT_SECRET.name(), secret);
		provider.instance(IInCoreDomains.class).setProperties(domain.uid, properties);

		if (AuthTypes.KERBEROS.name().equals(domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()))) {
			KerberosConfigHelper.createKeycloakKerberosConf(domain);
			KerberosConfigHelper.updateKrb5Conf();
		}

	}

	private void initExternal() {
		logger.info("Init external authentication config for domain {}", domain.uid);

		String opendIdHost = domain.value.properties.get(AuthDomainProperties.OPENID_HOST.name());

		JsonObject conf = getOpenIdConfiguration(opendIdHost);

		boolean somethingChanged = false;

		String key = AuthDomainProperties.OPENID_AUTHORISATION_ENDPOINT.name();
		String val = conf.getString("authorization_endpoint");
		somethingChanged = hasValueChanged(somethingChanged, key, val);

		key = AuthDomainProperties.OPENID_TOKEN_ENDPOINT.name();
		val = conf.getString("token_endpoint");
		somethingChanged = hasValueChanged(somethingChanged, key, val);

		key = AuthDomainProperties.OPENID_JWKS_URI.name();
		val = conf.getString("jwks_uri");
		somethingChanged = hasValueChanged(somethingChanged, key, val);

		key = AuthDomainProperties.OPENID_ISSUER.name();
		val = Optional.ofNullable(conf.getString("access_token_issuer")).orElse(conf.getString("issuer"));
		somethingChanged = hasValueChanged(somethingChanged, key, val);

		key = AuthDomainProperties.OPENID_END_SESSION_ENDPOINT.name();
		val = conf.getString("end_session_endpoint");
		somethingChanged = hasValueChanged(somethingChanged, key, val);

		logger.info("Domain propertie: {}", domain.value.properties);
		if (somethingChanged) {
			provider.instance(IInCoreDomains.class).setProperties(domain.uid, domain.value.properties);
		}

	}

	private JsonObject getOpenIdConfiguration(String openIdHost) {
		String configuration = SyncHttpClient.getInstance().get(openIdHost);
		return new JsonObject(configuration);
	}

	private boolean hasValueChanged(boolean somethingChanged, String key, String val) {
		if (val == null && domain.value.properties.get(key) != null) {
			domain.value.properties.remove(key);
			somethingChanged = true;
		} else if (val != null && !val.equals(domain.value.properties.get(key))) {
			domain.value.properties.put(key, val);
			somethingChanged = true;
		}
		return somethingChanged;
	}

	public void update() {
		if (domain.value.properties != null && AuthTypes.OPENID == AuthTypes
				.get(domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()))) {
			initExternal();
			ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IKeycloakAdmin.class)
					.deleteRealm(domain.uid);
			KerberosConfigHelper.updateGlobalRealmKerb();
			KerberosConfigHelper.updateKrb5Conf();
		} else {
			updateKeycloak();
		}
	}

	private void updateKeycloak() {
		logger.info("Update keycloak config for domain {}", domain.uid);

		String clientId = IKeycloakUids.clientId(IKeycloakUids.realmId(domain.uid));
		IKeycloakClientAdmin kcCientService = provider.instance(IKeycloakClientAdmin.class, domain.uid);

		OidcClient oc = kcCientService.getOidcClient(clientId);
		if (oc != null && !oc.secret.equals(provider.instance(IInternalCredentials.class, domain.uid)
				.getDomainCredentialById(AuthDomainProperties.OPENID_CLIENT_SECRET.name()))) {
			oc = null;
		}

		if (oc == null) {
			IKeycloakKerberosAdmin krbProv = provider.instance(IKeycloakKerberosAdmin.class,
					KeycloakManager.GLOBAL_VIRT);
			KerberosComponent krbComp = null;
			if (KeycloakManager.GLOBAL_VIRT.equals(domain.uid)) {
				krbComp = krbProv.getKerberosProvider(IKeycloakUids.kerberosComponentName(KeycloakManager.GLOBAL_VIRT));
			}

			provider.instance(IKeycloakAdmin.class).deleteRealm(domain.uid);
			initKeycloak();
			oc = kcCientService.getOidcClient(clientId);
			if (krbComp != null) {
				krbProv.create(krbComp);
			}
		}

		KeycloakUrls keycloakUrls = new KeycloakUrls(domain.uid);
		Set<String> currentUrls = keycloakUrls.getDomainUrls();
		if (!oc.redirectUris.containsAll(currentUrls) || !currentUrls.containsAll(oc.redirectUris)) {
			oc.redirectUris = currentUrls;
			oc.baseUrl = keycloakUrls.getExternalUrl();
			kcCientService.updateClient(clientId, oc);
			logger.info("Domain {} update : Urls changed : updated oidc client", domain.uid);
		} else {
			logger.debug("Domain {} update : Urls did not change (no need to update oidc client)", domain.uid);
		}

		KerberosConfigHelper.updateKeycloakKerberosConf(domain);
	}

	public void init(boolean deleteFirst) {
		if (deleteFirst) {
			provider.instance(IKeycloakAdmin.class).deleteRealm(domain.uid);
		}

		waitForKeycloak();
		if (domain.value.properties != null
				&& AuthTypes.OPENID.name().equals(domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()))) {
			initExternal();
		} else {
			initKeycloak();
		}

	}

	private void waitForKeycloak() {
		IKeycloakAdmin keycloakAdminService = provider.instance(IKeycloakAdmin.class);

		int nbRetries = 0;
		while (nbRetries < KEYCLOAK_WAIT_MAX_RETRIES) {
			try {
				keycloakAdminService.getRealm("master");
				return;
			} catch (Exception e) {
				// keycloak is not available yet
			}

			try {
				Thread.sleep(5000);
			} catch (InterruptedException e) {
			}
			nbRetries++;
		}

		throw new ServerFault("Wait for keycloak timed out (keycloak still not responding)");
	}
}
