/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

import io.vertx.core.http.HttpMethod;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.keycloak.api.IKeycloakAdmin;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.api.Realm;
import net.bluemind.keycloak.utils.KeycloakAdminClient;
import net.bluemind.keycloak.utils.adapters.RealmAdapter;
import net.bluemind.keycloak.utils.endpoint.KeycloakEndpoints;
import net.bluemind.role.api.BasicRoles;

public class KeycloakAdminService implements IKeycloakAdmin {
	private static final Logger logger = LoggerFactory.getLogger(KeycloakAdminService.class);

	private RBACManager rbacManager;
	private BmContext context;

	public KeycloakAdminService(BmContext context) {
		rbacManager = new RBACManager(context);
		this.context = context;
	}

	@Override
	public void createRealm(String domainUid) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);

		String realmId = IKeycloakUids.realmId(domainUid);
		logger.info("Create realm {} for domain {}", realmId, domainUid);

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance().execute(
				KeycloakEndpoints.realmsAdminEndpoint(), HttpMethod.POST, RealmAdapter.build(domainUid).toJson());

		try {
			response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (Exception e) {
			throw new ServerFault("Failed to create realm", e);
		}

		String realmAdminEndpoint = KeycloakEndpoints.realmAdminEndpoint(realmId);
		response = KeycloakAdminClient.getInstance().execute(realmAdminEndpoint + "/authentication/required-actions",
				HttpMethod.GET);

		try {
			response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS).getJsonArray("results").stream()
					.filter(Objects::nonNull).filter(JsonObject.class::isInstance).map(ra -> (JsonObject) ra)
					.filter(ra -> "VERIFY_PROFILE".equals(ra.getString("alias"))).map(ra -> ra.put("enabled", false))
					.findAny().ifPresent(ra -> {
						CompletableFuture<JsonObject> r = KeycloakAdminClient.getInstance().execute(
								realmAdminEndpoint + "/authentication/required-actions/VERIFY_PROFILE", HttpMethod.PUT,
								ra);
						try {
							r.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
						} catch (Exception e) {
							throw new ServerFault("Failed to disable VERIFY_PROFILE required action", e);
						}
					});
		} catch (Exception e) {
			throw new ServerFault("Failed to get realm required actions", e);
		}

	}

	@Override
	public void deleteRealm(String domainUid) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);

		String realmId = IKeycloakUids.realmId(domainUid);
		logger.info("Delete realm {} for domain {}", realmId, domainUid);
		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance()
				.execute(KeycloakEndpoints.realmAdminEndpoint(realmId), HttpMethod.DELETE);
		try {
			response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (Exception e) {
			throw new ServerFault("Failed to delete realm", e);
		}
	}

	@Override
	public List<Realm> allRealms() throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);

		logger.info("Get realms");

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance()
				.execute(KeycloakEndpoints.realmsAdminEndpoint(), HttpMethod.GET);

		JsonObject json;
		try {
			json = response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (Exception e) {
			throw new ServerFault("Failed fetch realms");
		}

		List<Realm> ret = new ArrayList<>();
		json.getJsonArray("results").forEach(realm -> ret.add(RealmAdapter.fromJson((JsonObject) realm)));
		return ret;
	}

	@Override
	public Realm getRealm(String domainUid) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);

		String realmId = IKeycloakUids.realmId(domainUid);
		logger.info("Get realm {} for domain {}", realmId, domainUid);

		CompletableFuture<JsonObject> response = KeycloakAdminClient.getInstance()
				.execute(KeycloakEndpoints.realmAdminEndpoint(realmId), HttpMethod.GET);

		JsonObject json;
		try {
			json = response.get(KeycloakAdminClient.TIMEOUT, TimeUnit.SECONDS);
		} catch (Exception e) {
			throw new ServerFault("Failed to get realm", e);
		}

		return RealmAdapter.fromJson(json);
	}

	@Override
	public TaskRef initForDomain(String domainUid, Boolean deleteFirst) throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);

		return context.provider().instance(ITasksManager.class).run(new BlockingServerTask() {
			@Override
			public void run(IServerTaskMonitor monitor) throws Exception {
				String realmId = IKeycloakUids.realmId(domainUid);
				monitor.begin(1, String.format("Init keycloak realm %s for domain %s ... ", realmId, domainUid));
				try {
					KeycloakManager.forDomain(domainUid).init(deleteFirst != null && deleteFirst);
					monitor.end(true, String.format("Init keycloak realm %s for domain %s done ", realmId, domainUid),
							"[]");
				} catch (Exception e) {
					logger.error(e.getMessage(), e);
					monitor.end(false,
							String.format("Failed to init keycloak realm %s for domain %s ", realmId, domainUid), "[]");
				}

			}
		});
	}

	@Override
	public TaskRef reconfigure() throws ServerFault {
		rbacManager.check(BasicRoles.ROLE_MANAGE_DOMAIN);
		return context.provider().instance(ITasksManager.class).run(new BlockingServerTask() {

			@Override
			public void run(IServerTaskMonitor monitor) throws Exception {
				IInCoreDomains domApi = context.su().provider().instance(IInCoreDomains.class);
				List<String> notGlobalDomains = domApi.allUnfiltered().stream().filter(ivd -> !ivd.value.global)
						.map(ivd -> ivd.uid).toList();
				monitor.begin(1.0 + notGlobalDomains.size(),
						String.format("Reset keycloak for %d domains", notGlobalDomains.size() + 1));
				List<Realm> existing = allRealms();
				int errors = 0;
				for (Realm r : existing) {
					if (!"master".equals(r.realm)) {
						monitor.log("Clear existing realm {}", r.realm);
						try {
							deleteRealm(r.realm);
						} catch (Exception e) {
							monitor.log(e.getMessage(), Level.ERROR);
							errors++;
						}
					}
				}
				if (!initOneDomain(monitor, "global.virt")) {
					errors++;
				}
				for (String domainUid : notGlobalDomains) {
					if (!initOneDomain(monitor, domainUid)) {
						errors++;
					}
				}
				if (errors == 0) {
					monitor.end(true, "Domains reconfigured", "[]");
				} else {
					monitor.end(false, "Reconfiguration is incomplete, " + errors + " error(s) detected", "[]");
				}
			}

			/**
			 * @param monitor
			 * @param domainUid
			 * @return true if successful
			 */
			private boolean initOneDomain(IServerTaskMonitor monitor, String domainUid) {
				boolean error = false;
				try {
					KeycloakManager.forDomain(domainUid).init(false);
				} catch (Exception e) {
					logger.error(e.getMessage(), e);
					monitor.log(e.getMessage(), Level.ERROR);
					error = true;
				}
				monitor.progress(1, "Initializing realm for domain " + domainUid + " " + (error ? "[FAILED]" : "[OK]"));
				return !error;
			}
		});
	}

}