/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.internal;

import java.util.Collections;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Objects;

import net.bluemind.core.api.auth.AuthDomainProperties;
import net.bluemind.core.api.auth.AuthTypes;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.MaintenanceOperation;
import net.bluemind.directory.service.IDirEntryRepairSupport;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.keycloak.api.BluemindProviderComponent;
import net.bluemind.keycloak.api.IKeycloakAdmin;
import net.bluemind.keycloak.api.IKeycloakBluemindProviderAdmin;
import net.bluemind.keycloak.api.IKeycloakClientAdmin;
import net.bluemind.keycloak.api.IKeycloakKerberosAdmin;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.api.KerberosComponent;
import net.bluemind.keycloak.api.OidcClient;
import net.bluemind.keycloak.api.Realm;
import net.bluemind.keycloak.utils.adapters.BlueMindComponentAdapter;
import net.bluemind.keycloak.utils.adapters.KerberosComponentAdapter;
import net.bluemind.keycloak.utils.adapters.OidcClientAdapter;
import net.bluemind.keycloak.utils.adapters.RealmAdapter;
import net.bluemind.system.api.IInternalCredentials;

public class KeycloakRealmRepairSupport implements IDirEntryRepairSupport {
	private static final Logger logger = LoggerFactory.getLogger(KeycloakRealmRepairSupport.class);

	public static final MaintenanceOperation keycloakRepair = MaintenanceOperation.create("domain.keycloak",
			"Ensure that domain has a correct keycloak configuration (realm, client)");

	public static class Factory implements IDirEntryRepairSupport.Factory {
		@Override
		public IDirEntryRepairSupport create(BmContext context) {
			return new KeycloakRealmRepairSupport(context);
		}
	}

	public KeycloakRealmRepairSupport(BmContext context) {
	}

	@Override
	public Set<MaintenanceOperation> availableOperations(Kind kind) {
		if (kind == Kind.DOMAIN) {
			return Set.of(keycloakRepair);
		}
		return Collections.emptySet();
	}

	@Override
	public Set<InternalMaintenanceOperation> ops(Kind kind) {
		if (kind == Kind.DOMAIN) {
			return Set.of(new KeycloakRepairImpl());
		}
		return Collections.emptySet();

	}

	private static class KeycloakRepairImpl extends InternalMaintenanceOperation {
		private static class KeycloakConf {
			public static KeycloakConf build(String domainUid) {
				ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
				ItemValue<Domain> domain = provider.instance(IInCoreDomains.class).getUnfiltered(domainUid);

				Realm realm = provider.instance(IKeycloakAdmin.class, domainUid).getRealm(domainUid);

				if (realm == null) {
					return new KeycloakConf(domain, realm, null, null, null);
				}

				OidcClient oidcClient = provider.instance(IKeycloakClientAdmin.class, domainUid)
						.getOidcClient(IKeycloakUids.clientId(realm.id));
				BluemindProviderComponent bluemindProviderComponent = provider
						.instance(IKeycloakBluemindProviderAdmin.class, domainUid)
						.getBluemindProvider(IKeycloakUids.bmProviderId(realm.id));
				KerberosComponent kerberosComponent = provider.instance(IKeycloakKerberosAdmin.class, domainUid)
						.getKerberosProvider(IKeycloakUids.kerberosComponentName(realm.id));

				return new KeycloakConf(domain, realm, oidcClient, bluemindProviderComponent, kerberosComponent);
			}

			private ItemValue<Domain> domain;
			private Realm realm;
			private OidcClient oidcClient;
			private BluemindProviderComponent bluemindProviderComponent;
			private KerberosComponent kerberosComponent;

			private KeycloakConf(ItemValue<Domain> domain, Realm realm, OidcClient oidcClient,
					BluemindProviderComponent bluemindProviderComponent, KerberosComponent kerberosComponent) {
				this.domain = domain;
				this.realm = realm;
				this.oidcClient = oidcClient;
				this.bluemindProviderComponent = bluemindProviderComponent;
				this.kerberosComponent = kerberosComponent;
			}

			public boolean isOk() {
				Realm expectedRealm = RealmAdapter.build(domain.uid).realm;
				if (!Objects.equal(expectedRealm, realm)) {
					logger.error("Realm ko for domain: {} - must be:\n{}\nis: {}", domain.uid, expectedRealm, realm);
					return false;
				}

				OidcClient expectedOidcClient = OidcClientAdapter.build(domain.uid,
						IKeycloakUids.clientId(expectedRealm.id), Optional.empty()).oidcClient;
				expectedOidcClient.secret = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
						.instance(IInternalCredentials.class, domain.uid)
						.getDomainCredentialById(AuthDomainProperties.OPENID_CLIENT_SECRET.name());
				if (!Objects.equal(expectedOidcClient, oidcClient)) {
					logger.info("OidcClient ko for domain: {} - must be:\n{}\nis: {}", domain.uid, expectedOidcClient,
							oidcClient);
					return false;
				}

				BluemindProviderComponent expectedBlueMindProviderComponent = expectedBluemindProviderComponent();
				// Can't be checked as Keycloak keep it secret
				expectedBlueMindProviderComponent.bmCoreToken = null;
				if (!Objects.equal(expectedBlueMindProviderComponent, bluemindProviderComponent)) {
					logger.info("BluemindProvider ko for domain: {} - must be:\n{}\nis: {}", domain.uid,
							expectedBlueMindProviderComponent, bluemindProviderComponent);
					return false;
				}

				if (AuthTypes.KERBEROS.name()
						.equals(domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()))) {
					KerberosComponent expectedKerberosComponent = KerberosComponentAdapter.build(domain).component;
					if (!Objects.equal(expectedKerberosComponent, kerberosComponent)) {
						logger.info("KerberosProvider ko for domain: {} - must be:\n{}\nis: {}", domain.uid,
								expectedKerberosComponent, kerberosComponent);
						return false;
					}
				}

				return true;
			}

			public BluemindProviderComponent expectedBluemindProviderComponent() {
				return BlueMindComponentAdapter.build(domain.uid).component;
			}
		}

		public KeycloakRepairImpl() {
			super(keycloakRepair.identifier, null, null, 1);
		}

		@Override
		public void check(String domainUid, DirEntry entry, RepairTaskMonitor monitor) {
			monitor.begin(1, "Check Keycloak configuration for domain: " + domainUid);

			boolean confOk = KeycloakConf.build(domainUid).isOk();

			monitor.progress(1, "Keycloak configuration is " + (confOk ? "ok" : "ko"));
			monitor.end();
		}

		@Override
		public void repair(String domainUid, DirEntry entry, RepairTaskMonitor monitor) {
			monitor.begin(2, "Check Keycloak configuration for domain: " + domainUid);
			KeycloakConf keycloakConf = KeycloakConf.build(domainUid);

			if (keycloakConf.isOk()) {
				monitor.progress(1, "Keycloak configuration don't need repair for domain: " + domainUid);

				ComponentService.forDomain(domainUid).updateComponent(
						new BlueMindComponentAdapter(keycloakConf.expectedBluemindProviderComponent()).toJson());
				monitor.progress(1, "Keycloak BlueMind provider updated for domain: " + domainUid);
				monitor.end();
				return;
			}

			monitor.progress(1, "Keycloak configuration need repair for domain: " + domainUid);

			KeycloakManager.forDomain(domainUid).init(true);

			monitor.progress(1, "Reset keycloak configuration for domain: " + domainUid);
			monitor.end();
		}
	}
}
