/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.indexing.incremental;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;

import net.bluemind.core.container.api.IChangelogSupport;
import net.bluemind.core.container.api.IReadByIdSupport;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore.ContainerSyncState;

public abstract class AbstractTypeIndexer<T> implements TypeIndexerFactory.TypeIndexer {

	private static final Logger logger = LoggerFactory.getLogger(AbstractTypeIndexer.class);

	public interface IndexWriter<W> {
		void write(ItemValue<W> toIndex);

		void deleteByIds(List<Long> itemIds);

	}

	private final IChangelogSupport clSupport;
	private final IReadByIdSupport<T> crud;
	private final IndexWriter<T> indexer;

	private static final ExecutorService vtPool = Executors.newVirtualThreadPerTaskExecutor();

	protected AbstractTypeIndexer(IChangelogSupport clSupport, IReadByIdSupport<T> crud, IndexWriter<T> indexer) {
		this.clSupport = clSupport;
		this.crud = crud;
		this.indexer = indexer;
	}

	@Override
	public long indexDelta(ContainerSyncState startState) {
		if (startState.version() == IIncrementalIndexingStore.DO_NOT_INDEX) {
			return startState.version();
		}

		try {
			var chrono = Stopwatch.createStarted();
			ContainerChangeset<Long> changeset = clSupport.changesetById(startState.version());
			List<Long> itemIdsUpdatedCreated = new ArrayList<>(changeset.created.size() + changeset.updated.size());
			itemIdsUpdatedCreated.addAll(changeset.created);
			itemIdsUpdatedCreated.addAll(changeset.updated);

			List<List<Long>> itemIdsUpdatedCreatedPartitions;
			if (!itemIdsUpdatedCreated.isEmpty()) {
				itemIdsUpdatedCreatedPartitions = Lists.partition(itemIdsUpdatedCreated, 200);
			} else {
				itemIdsUpdatedCreatedPartitions = Collections.emptyList();
			}

			List<List<Long>> itemIdsDeletedPartitions;
			if (changeset.deleted != null && !changeset.deleted.isEmpty()) {
				itemIdsDeletedPartitions = Lists.partition(changeset.deleted, 1000);
			} else {
				itemIdsDeletedPartitions = Collections.emptyList();
			}

			int totalPartitionCount = itemIdsDeletedPartitions.size() + itemIdsUpdatedCreatedPartitions.size();
			if (totalPartitionCount == 0) {
				// Nothing to do ?
				return changeset.version;
			}

			CompletableFuture<?>[] futures = new CompletableFuture[totalPartitionCount];
			int futureIdx = 0;
			if (!itemIdsUpdatedCreatedPartitions.isEmpty()) {
				for (var partition : itemIdsUpdatedCreatedPartitions) {
					futures[futureIdx++] = CompletableFuture
							.runAsync(() -> crud.multipleGetById(partition).forEach(indexer::write), vtPool);
				}
			}
			if (!itemIdsDeletedPartitions.isEmpty()) {
				for (var partition : itemIdsDeletedPartitions) {
					futures[futureIdx++] = CompletableFuture.runAsync(() -> indexer.deleteByIds(partition), vtPool);
				}
			}
			CompletableFuture.allOf(futures).orTimeout(10, TimeUnit.MINUTES).join();
			if (logger.isInfoEnabled() && changeset.version > startState.version()) {
				logger.info("[{}] incr update from v{} to v{} took {}ms (c: {}, u: {}, d: {})",
						startState.cont().value(), startState.version(), changeset.version,
						chrono.elapsed(TimeUnit.MILLISECONDS), changeset.created.size(), changeset.updated.size(),
						changeset.deleted.size());
			}
			return changeset.version;
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return startState.version();
		}
	}

}
